<?php declare(strict_types=1);

namespace NIMIUS\Graphing\Graph\Behaviour;

use NIMIUS\Graphing\Canvas\Drawable;
use NIMIUS\Graphing\Canvas\Drawer;
use NIMIUS\Graphing\Color;
use NIMIUS\Graphing\Series;

class SeriesDrawer implements Drawable
{
    /** @var Series */
    protected $series;

    /** @var Color */
    protected $fillColor;

    /** @var Color|null */
    protected $strokeColor;

    /** @var int */
    protected $strokeWidth;

    /** @var bool */
    protected $autoscale;

    public function __construct(
        Series $series,
        Color $fillColor,
        ?Color $strokeColor,
        int $strokeWidth = 1,
        bool $autoscale = true
    ) {
        $this->series = $series;
        $this->fillColor = $fillColor;
        $this->strokeColor = $strokeColor;
        $this->strokeWidth = $strokeWidth;
        $this->autoscale = $autoscale;
    }

    public function draw(Drawer $drawer): void
    {
        // Figure out bounds for later calculations.
        $points = $this->series->getPoints();
        $xBounds = $this->series->xBounds();
        $xDelta = max(1, abs($xBounds[1] - $xBounds[0]));
        $yBounds = $this->series->yBounds();
        if ($this->autoscale) {
            $yBounds[0] -= $yBounds[0] * .1;
        } else {
            $yBounds[0] = 0;
        }
        $yDelta = max(1, abs($yBounds[1] - $yBounds[0]));

        $dataPoints = [
            [ 100, 100 ], // Bottom right corner
            [ 0, 100 ], // Bottom left corner
        ];

        // Resize bounds in order to fit in a range from 0 - 100.
        foreach ($points as $point) {
            $dataPoints[] = [
                (($point[0] / $xDelta) - $xBounds[0]) * 100,
                (1 - ($point[1] - $yBounds[0]) / $yDelta) * 100,
            ];
        }

        $drawer->filledPolygon($dataPoints, $this->fillColor);

        if ($this->strokeWidth && $this->strokeColor) {
            $drawer->strokedPolygon(array_slice($dataPoints, 2), $this->strokeColor, $this->strokeWidth);
        }
    }

    public function drawableSize(array $sizeLeftInImage): array
    {
        return $sizeLeftInImage;
    }
}
