<?php declare(strict_types=1);

namespace NIMIUS\Graphing;

use NIMIUS\Graphing\Exception\InvalidHexStringException;

/**
 * Represents a color using it's red, green, blue and alpha components.
 * All of the components can be set from 0 - 255
 */
class Color
{
    protected $red;
    protected $green;
    protected $blue;
    protected $alpha;

    /**
     * Creates a new color from the given hex string.
     * Note: RGBA colors are not supported by this method.
     *
     * @param string $hex
     * @return static
     * @throws InvalidHexStringException
     */
    public static function hex(string $hex): self
    {
        if (substr($hex, 0, 1) === '#') {
            $hex = substr($hex, 1);
        }

        // Convert short notation to long notation
        if (\strlen($hex) === 3 || \strlen($hex) === 4) {
            $length = strlen($hex);
            $original = $hex;
            $hex = '';
            for ($i = 0; $i < $length; $i++) {
                $hex .= str_repeat(substr($original, $i, 1), 2);
            }
        }

        if (\strlen($hex) === 6) {
            // Create new color without alpha.
            [ $r, $g, $b ] = sscanf($hex, '%02x%02x%02x');
            return new static($r, $g, $b);
        } elseif (\strlen($hex) === 8) {
            // Create new color with alpha.
            [ $r, $g, $b, $a ] = sscanf($hex, '%02x%02x%02x%02x');
            return new static($r, $g, $b, $a);
        }

        throw new InvalidHexStringException('The given hex string is not valid');
    }

    /**
     * @param int $red
     * @param int $green
     * @param int $blue
     * @param int|null $alpha
     */
    public function __construct(int $red, int $green, int $blue, int $alpha = null)
    {
        $this->red = $red;
        $this->green = $green;
        $this->blue = $blue;
        $this->alpha = $alpha;
    }

    /**
     * Allocates the current color on the given resource
     * and returns the allocation id.
     *
     * @param resource $resource
     * @return int
     */
    public function allocate($resource)
    {
        if ($this->alpha) {
            return imagecolorallocatealpha(
                $resource,
                $this->red,
                $this->green,
                $this->blue,
                (int) ((1 - $this->alpha / 255) * 127)
            );
        }

        return imagecolorallocate(
            $resource,
            $this->red,
            $this->green,
            $this->blue
        );
    }
}
