<?php declare(strict_types=1);

namespace NIMIUS\Graphing\Canvas;

use NIMIUS\Graphing\Exception\GdException;

/**
 * Canvas on which an image can be drawn.
 */
class Canvas
{

    /**
     * @var resource
     */
    protected $resource;

    /**
     * @var int[]
     */
    protected $size;

    public function __construct(array $size)
    {
        $this->size = $size;

        $resource = imagecreatetruecolor($size[0], $size[1]);
        if ($resource === false) {
            throw new GdException('Could not initialize true color image');
        }
        $this->resource = $resource;

        $transparent = imagecolorallocatealpha($this->resource, 0, 0, 0, 127);
        imagefill($this->resource, 0, 0, $transparent);
        imagesavealpha($this->resource, true);
        imageantialias($this->resource, true);
    }

    /**
     * Adds a new layer to the current canvas.
     *
     * @param Drawable $drawable
     * @param array $position
     * @param array|null $size
     * @return Canvas
     */
    public function addLayer(Drawable $drawable, array $position, array $size = null): self
    {
        $sizeLeftInImage = [ $this->size[0] - $position[0], $this->size[1] - $position[1] ];
        $size = $size ?: $drawable->drawableSize($sizeLeftInImage);
        $drawer = new Drawer($this->resource, $position, $size);
        $drawable->draw($drawer);

        return $this;
    }

    /**
     * Saves the current canvas as a PNG.
     *
     * @param string $path
     * @return Canvas
     */
    public function savePng(string $path): self
    {
        imagepng($this->resource, $path);
        return $this;
    }
}
