<?php declare(strict_types = 1);

namespace NIMIUS\Graphing\Graph\Behaviour;

use NIMIUS\Graphing\Direction;

abstract class AbstractAxis
{

    public static function calculateTicks(array $dataBounds, int $numberOfTicks, string $direction, string $format = '%d'): array
    {

        $delta = max(1, $dataBounds[1] - $dataBounds[0]);
        $ticksEvery = floor($delta / $numberOfTicks);
        $ticksEveryPercent = ($ticksEvery / $delta) * 100;

        $positions = [];
        for ($i = 0; $i <= $numberOfTicks; $i++) {
            $label = sprintf($format, round($dataBounds[0] + $i * $ticksEvery));

            $x = 0;
            $y = 0;
            if ($direction === Direction::HORIZONTAL) {
                $x = $i * $ticksEveryPercent;
            } else {
                $y = 100 - ($i * $ticksEveryPercent);
            }

            $positions[] = [ $x, $y, $label ];
        }

        return $positions;
    }

    /** @var array */
    protected $dataBounds;

    /** @var int */
    protected $numberOfTicks;

    /** @var string */
    protected $format = '%d';

    /** @var string */
    protected $direction;

    public function __construct(array $dataBounds, string $direction, int $numberOfTicks = 10)
    {
        $this->dataBounds = $dataBounds;
        $this->numberOfTicks = $numberOfTicks;
        $this->direction = $direction;
    }

    public function setFormat(string $format): self
    {
        $this->format = $format;
        return $this;
    }

    protected function getTickPositions(): array
    {
        return static::calculateTicks($this->dataBounds, $this->numberOfTicks, $this->direction, $this->format);
    }

    public function drawableSize(array $sizeLeftInImage): array
    {
        return $sizeLeftInImage;
    }
}
