<?php

namespace NIMIUS\Graphing\Graph;

use NIMIUS\Graphing\Canvas\Drawable;
use NIMIUS\Graphing\Canvas\Drawer;
use NIMIUS\Graphing\Color;
use NIMIUS\Graphing\Graph\Behaviour\HorizontalAxisWithLabels;
use NIMIUS\Graphing\Graph\Behaviour\VerticalAxisWithLabels;
use NIMIUS\Graphing\Series;

/**
 * Simple filled line graph.
 * This graph displays the given Series as a filled area with an outline.
 *
 * The setters can be used in order to customize the rendering of the graph.
 * The following attributes can be customized:
 *
 * - `verticalAxis`: Sets the color of the vertical axis. The axis will not be rendered if
 *                  this attribute is set to null.
 * - `verticalTicks`: The number of ticks / labels that exist on the vertical axis. Defaults to 10.
 * - `horizontalAxis`: Sets the color of the horizontal axis. The axis will not be rendered if
 *                     this attribute is set to null.
 * - `horizontalTicks`: The number of ticks / labels that exist on the horizontal axis. Defaults to 10.
 * - `fillColor`: Color that is used to fill the area.
 * - `strokeColor`: Color that is used to stroke the area.
 * - `autoScale`: Whether or not to automatically scale the lower bound of the y-axis.
 *                If enabled the lowest point on the y-axis will correspond to the lowest point
 *                of the data. If disabled the y-axis will start at 0.
 *
 * @example
 * $lastPoint = $this->points[0];
 * $totalDistance = 0;
 *
 * $series = new Series();
 * foreach ($this->points as $point) {
 *      $totalDistance += $point->calculateDistance($lastPoint);
 *      $series->addPoint($totalDistance, $point->elevation);
 *      $lastPoint = $point;
 * }
 *
 * $graph = new Graph\FilledLineGraph($series);
 * (new Canvas([ $width * 2, $height * 2 ]))
 *      ->addLayer($graph, [ 0, 0 ])
 *      ->savePng(base_path('test2.png'));
 */
class FilledLineGraph implements Drawable
{

    /**
     * @var Series
     */
    protected $series;

    /**
     * @var Color
     */
    protected $fillColor;

    /**
     * @var Color
     */
    protected $strokeColor;

    /**
     * @var Color
     */
    protected $horizontalAxis;

    /**
     * @var int
     */
    protected $horizontalTicks = 10;

    /**
     * @var Color
     */
    protected $verticalAxis;

    /**
     * @var int
     */
    protected $verticalTicks = 10;

    /**
     * @var bool
     */
    protected $autoScale = true;

    /**
     * @param Series $series
     * @throws \NIMIUS\Graphing\Exception\InvalidHexStringException
     */
    public function __construct(Series $series)
    {
        $this->series = $series;
        $this->fillColor = Color::hex('#3097d199');
        $this->strokeColor = Color::hex('#0067a1');
        $this->verticalAxis = Color::hex('#111');
        $this->horizontalAxis = Color::hex('#111');
    }


    /**
     * Draws the drawable using the given drawer.
     * This method applies the correct drawing instructions
     * in order for the current object to be visible on screen.
     *
     * @param Drawer $drawer
     */
    public function draw(Drawer $drawer)
    {
        // Figure out bounds for later calculations.
        $points = $this->series->getPoints();
        $xBounds = $this->series->xBounds();
        $xDelta = abs($xBounds[1] - $xBounds[0]);
        $yBounds = $this->series->yBounds();
        if ($this->autoScale) {
            $yBounds[0] -= $yBounds[0] * .1;
        } else {
            $yBounds[0] = 0;
        }
        $yDelta = abs($yBounds[1] - $yBounds[0]);

        $dataPoints = [
            [100, 100], // Bottom right corner
            [0, 100], // Bottom left corner
        ];

        // Resize bounds in order to fit in a range from 0 - 100.
        foreach ($points as $point) {
            $dataPoints[] = [
                (($point[0] / $xDelta) - $xBounds[0]) * 100,
                (1 - ($point[1] - $yBounds[0]) / $yDelta) * 100
            ];
        }

        /*
         * Figure out the size of the area that is used to draw the
         * polygon portion of the graph.
         * This depends on the combination of axis that are being used
         * since we have to make some place in order to draw the axis.
         */
        $shiftForAxis = [0, 0];
        $sizeChangeForAxis = [0, 0];
        $verticalAxisWidth = 5;
        $horizontalOffsetForVerticalAxis = 2;
        $horizontalAxisHeight = 7;
        $verticalOffsetForHorizontalAxis = 2;
        if ($this->horizontalAxis) {
            $sizeChangeForAxis[0] -= $verticalOffsetForHorizontalAxis;
            $sizeChangeForAxis[1] -= $horizontalAxisHeight;
            if (!$this->verticalAxis) {
                $shiftForAxis[0] += $verticalOffsetForHorizontalAxis;
                $sizeChangeForAxis[0] -= $verticalOffsetForHorizontalAxis;
            }
        }
        if ($this->verticalAxis) {
            $shiftForAxis[0] += $verticalAxisWidth;
            $sizeChangeForAxis[0] -= $verticalAxisWidth;
            $shiftForAxis[1] += $horizontalOffsetForVerticalAxis;
            $sizeChangeForAxis[1] -= $this->horizontalAxis ? $horizontalOffsetForVerticalAxis : $horizontalOffsetForVerticalAxis * 2;
        }

        // Draw polygon (filled and line)
        $graphDrawer = $drawer->shiftDrawingCanvas($shiftForAxis, [ 100 + $sizeChangeForAxis[0], 100 + $sizeChangeForAxis[1] ]);
        $graphDrawer
            ->filledPolygon($dataPoints, $this->fillColor)
            ->strokedPolygon($dataPoints, $this->strokeColor);

        // Add horizontal axis.
        if ($this->horizontalAxis) {
            $horizontalAxis = new HorizontalAxisWithLabels(
                $xBounds,
                $this->horizontalTicks,
                $this->horizontalAxis
            );
            $horizontalAxis->draw(
                $drawer->shiftDrawingCanvas(
                    [$shiftForAxis[0], 100 - $horizontalAxisHeight],
                    [ 100 + $sizeChangeForAxis[0], $horizontalAxisHeight ]
                )
            );
        }

        // Add vertical axis.
        if ($this->verticalAxis) {
            $verticalAxis = new VerticalAxisWithLabels(
                $yBounds,
                $this->verticalTicks,
                $this->verticalAxis
            );
            $verticalAxis->draw(
                $drawer->shiftDrawingCanvas(
                    [0, $horizontalOffsetForVerticalAxis],
                    [ $verticalAxisWidth, 100 + $sizeChangeForAxis[1]]
                )
            );
        }
    }

    /**
     * Returns an array of how large the drawable thinks itself is.
     * These sizes will be used if no explicit size has been specified.
     *
     * The returned size must be an array of [ width, height ].
     *
     * @param array $sizeLeftInImage - The size that is left in the image.
     *                                 This should be the maximum size.
     * @return array
     */
    public function drawableSize(array $sizeLeftInImage): array
    {
        return $sizeLeftInImage;
    }

    /**
     * @param Color $fillColor
     * @return FilledLineGraph
     */
    public function setFillColor(Color $fillColor): self
    {
        $this->fillColor = $fillColor;
        return $this;
    }

    /**
     * @param Color $strokeColor
     * @return FilledLineGraph
     */
    public function setStrokeColor(Color $strokeColor): self
    {
        $this->strokeColor = $strokeColor;
        return $this;
    }

    /**
     * @param Color $horizontalAxis
     * @return FilledLineGraph
     */
    public function setHorizontalAxis(Color $horizontalAxis = null): self
    {
        $this->horizontalAxis = $horizontalAxis;
        return $this;
    }

    /**
     * @param Color $verticalAxis
     * @return FilledLineGraph
     */
    public function setVerticalAxis(Color $verticalAxis = null): self
    {
        $this->verticalAxis = $verticalAxis;
        return $this;
    }

    /**
     * @param bool $autoScale
     * @return FilledLineGraph
     */
    public function setAutoScale(bool $autoScale): self
    {
        $this->autoScale = $autoScale;
        return $this;
    }

    /**
     * @param int $horizontalTicks
     * @return FilledLineGraph
     */
    public function setHorizontalTicks(int $horizontalTicks): self
    {
        $this->horizontalTicks = $horizontalTicks;
        return $this;
    }

    /**
     * @param int $verticalTicks
     * @return FilledLineGraph
     */
    public function setVerticalTicks(int $verticalTicks): self
    {
        $this->verticalTicks = $verticalTicks;
        return $this;
    }
}