<?php
namespace NIMIUS\Graphing\Graph\Behaviour;

use NIMIUS\Graphing\Canvas\Drawable;
use NIMIUS\Graphing\Canvas\Drawer;
use NIMIUS\Graphing\Color;

/**
 * Draws a horizontal axis with labels.
 *
 * This is meant to be used as shared behaviour for all charts graphs
 * that use linear axis.
 *
 * @example
 * // Add simple axis to the bottom 10% of the drawing canvas.
 * $xAxis = new HorizontalAxisWithLabel(
 *      $series->xBounds()
 * );
 * $xAxis->draw(
 *      $drawer->shiftDrawingCanvas([ 0, 90 ])
 * );
 *
 * @example
 * // Same as above but with 15 ticks and in red
 * $xAxis = new HorizontalAxisWithLabel(
 *      $series->xBounds(),
 *      15,
 *      new Color(255, 0, 0)
 * );
 * $xAxis->draw(
 *      $drawer->shiftDrawingCanvas([ 0, 90 ])
 * );
 */
class HorizontalAxisWithLabels implements Drawable
{

    /**
     * @var array
     */
    protected $dataBounds;

    /**
     * @var int
     */
    protected $numberOfTicks;

    /**
     * @var Color
     */
    protected $color;

    /**
     * @param array $dataBounds
     * @param int $numberOfTicks
     * @param Color|null $color
     */
    public function __construct(array $dataBounds, int $numberOfTicks = 10, Color $color = null)
    {
        $this->dataBounds = $dataBounds;
        $this->numberOfTicks = $numberOfTicks;
        $this->color = $color ?: new Color(0, 0, 0);
    }

    /**
     * Draws the drawable using the given drawer.
     * This method applies the correct drawing instructions
     * in order for the current object to be visible on screen.
     *
     * @param Drawer $drawer
     */
    public function draw(Drawer $drawer)
    {
        // Axis line
        $drawer->line([ 0, 0 ], [ 100, 0 ], $this->color);

        // Ticks and labels
        $delta = $this->dataBounds[1] - $this->dataBounds[0];
        $ticksEvery = floor($delta / $this->numberOfTicks);
        $ticksEveryPercent = ($ticksEvery / $delta) * 100;
        for ($i = 0; $i <= $this->numberOfTicks; $i++) {
            $x = $i * $ticksEveryPercent;
            $text = $this->dataBounds[0] + $i * $ticksEvery;

            $drawer
                ->line([$x, 0], [$x, 30], $this->color)
                ->text(
                    $text,
                    [$x, 90],
                     $this->color,
                     [
                         Drawer::TEXT_ALIGN => Drawer::VERTICAL_ALIGN_CENTER,
                         Drawer::FONT_SIZE => 55
                     ]
                );
        }
    }

    /**
     * Returns an array of how large the drawable thinks itself is.
     * These sizes will be used if no explicit size has been specified.
     *
     * The returned size must be an array of [ width, height ].
     *
     * @param array $sizeLeftInImage - The size that is left in the image.
     *                                 This should be the maximum size.
     * @return array
     */
    public function drawableSize(array $sizeLeftInImage): array
    {
        return $sizeLeftInImage;
    }
}