# Graphing

Simple Library for rendering graphs into static images.
The current implementation only supports filled line graphs but
is built in a way that makes extension easier.

![Example Graph](./resources/readme/example.png)

```php
// Build data series out of points.
$series = new Series();
foreach ($points as $point) {
    $series->addPoint(
        $point->distance,
        $point->elevation
    );
}

// Draw graph.
$graph = new FilledLineGraph($series);
(new Canvas([ 1600, 600 ]))
    ->addLayer($graph, [ 0, 0 ])
    ->savePng($path);
```

## Internals
In order to make extension easier a layer of Abstraction has been added on top of PHP's
native GD library: Canvas & Drawables.

In their Core they are exactly what they sound like: A canvas can be painted on and a Drawable
can be painted onto a canvas. A canvas does not need to know about the implementation details of
each drawable and a Drawable does not need to know anything about the canvas it is painted on.

To maintain this level of separation a Drawable does not work with absolute units: A `Drawer` is
passed to a Drawable which expects all values passed to it to be percentages. The `Drawer` then
converts these values into real pixels. This way the Drawer is coupled to the Canvas but the Canvas
must not worry about translating content (in order to draw drawables at a cetain position) or even
the details of how the GD Library works and Drawables are fully decoupled from the Canvas they are
painted on.

The following shows an example implementation of a drawable:

```php
class Rectangle implements Drawable {
    
    public function draw(Drawer $drawer) {
        // Because drawables must not be concerned about their actual size
        // it is enough to just paint the full area given to us.
        $drawer->filledPolygon(
            [
                [   0,   0 ], // Upper right corner ( 0% left, 0% top )
                [ 100,   0 ], // Upper right corner ( 100% left, 0% top )
                [ 100, 100 ], // Lower right corner ( 100% left, 100% top )
                [   0, 100 ], // Lower left corner ( 0% left, 100% right )
            ],
            Color::hex('#000')
        ); 
    }

    /**
     * If no size is specified render a rectangle at 200x200 pixels.
     * @return array
     */
    public function drawableSize(array $sizeLeftInImage): array
    {
        return [ 200, 200 ];
    }

}
```