<?php
/**
 * Created by PhpStorm.
 * User: j
 * Date: 7/27/18
 * Time: 10:30 AM
 */

namespace NIMIUS\Graphing\Graph\Behaviour;


use NIMIUS\Graphing\Canvas\Drawable;
use NIMIUS\Graphing\Canvas\Drawer;
use NIMIUS\Graphing\Color;


/**
 * Draws a vertical axis with labels.
 *
 * This is meant to be used as shared behaviour for all charts graphs
 * that use linear axis.
 *
 * @example
 * // Add simple axis to the left 10% of the drawing canvas.
 * $yAxis = new VerticalAxisWithLabels(
 *      $series->yBounds()
 * );
 * $yAxis->draw(
 *      $drawer->shiftDrawingCanvas([ 0, 0 ], [ 10, 100 ])
 * );
 *
 * @example
 * // Same as above but with 15 ticks and in red
 * $yAxis = new VerticalAxisWithLabels(
 *      $series->yBounds(),
 *      15,
 *      new Color(255, 0, 0)
 * );
 * $yAxis->draw(
 *      $drawer->shiftDrawingCanvas([ 0, 0 ], [ 10, 100 ])
 * );
 */
class VerticalAxisWithLabels implements Drawable
{

    /**
     * @var array
     */
    protected $dataBounds;

    /**
     * @var int
     */
    protected $ticksEvery;

    /**
     * @var Color
     */
    protected $color;

    /**
     * @param array $dataBounds
     * @param int $ticksEvery
     * @param Color|null $color
     */
    public function __construct(array $dataBounds, int $ticksEvery = 10, Color $color = null)
    {
        $this->dataBounds = $dataBounds;
        $this->ticksEvery = $ticksEvery;
        $this->color = $color ?: new Color(0, 0, 0);
    }

    /**
     * Draws the drawable using the given drawer.
     * This method applies the correct drawing instructions
     * in order for the current object to be visible on screen.
     *
     * @param Drawer $drawer
     */
    public function draw(Drawer $drawer)
    {
        // Axis line
        $drawer->line([ 100, 0 ], [ 100, 100 ], $this->color);

        // Ticks and labels
        $delta = $this->dataBounds[1] - $this->dataBounds[0];
        $ticksEveryUnit = round($delta / $this->ticksEvery);
        $ticksEveryPercent = round(($ticksEveryUnit / $delta) * 100);
        $numberOfTicks = floor(100 / $ticksEveryPercent);
        for ($i = 0; $i <= $numberOfTicks; $i++) {
            $y = 100 - ($i * $ticksEveryPercent);
            $text = round($this->dataBounds[1] + $i * $ticksEveryPercent);
            $drawer
                ->line([ 80, $y ], [ 100, $y], $this->color)
                ->text(
                    $text,
                    [ 75, $y ],
                    $this->color,
                    [
                        Drawer::TEXT_ALIGN => Drawer::TEXT_ALIGN_RIGHT,
                        Drawer::VERTICAL_ALIGN => Drawer::VERTICAL_ALIGN_CENTER,
                        Drawer::FONT_SIZE => 4
                    ]
                );
        }
    }

    /**
     * Returns an array of how large the drawable thinks itself is.
     * These sizes will be used if no explicit size has been specified.
     *
     * The returned size must be an array of [ width, height ].
     *
     * @param array $sizeLeftInImage - The size that is left in the image.
     *                                 This should be the maximum size.
     * @return array
     */
    public function drawableSize(array $sizeLeftInImage): array
    {
        return $sizeLeftInImage;
    }
}