<?php
namespace NIMIUS\Graphing\Canvas;
use NIMIUS\Graphing\Color;

/**
 * Intermediate Object between a canvas and it's drawables.
 * This does two things:
 * - Encapsulates all of GDs drawing methods in an easier API
 * - Gives drawables an easy way to draw things without having to worry
 *   about offsets and sizes: All sizes given to the drawer are between 0 and 100
 *   with the drawer being in charge of calculating real coordinates out of those
 *   percentages.
 */
class Drawer
{
    const FONT_SIZE = 'fontSize';
    const ANGLE = 'angle';
    const TTF_FONT = 'font';
    const TEXT_ALIGN = 'textAlign';
    const TEXT_ALIGN_LEFT = 'left';
    const TEXT_ALIGN_CENTER = 'center';
    const TEXT_ALIGN_RIGHT = 'right';
    const VERTICAL_ALIGN = 'verticalAlign';
    const VERTICAL_ALIGN_BOTTOM = 'bottom';
    const VERTICAL_ALIGN_CENTER = 'center';
    const VERTICAL_ALIGN_TOP = 'top';

    /**
     * @var \resource
     */
    protected $resource;

    /**
     * @var int[]
     */
    protected $offset;

    /**
     * @var int[]
     */
    protected $size;

    public function __construct($resource, $offset = [ 0, 0 ], $size = [ 200, 200 ])
    {
        $this->resource = $resource;
        $this->offset = $offset;
        $this->size = $size;
    }

    /**
     * Draws a line from the given start coordinates to the given end coordinates
     * in the given color.
     *
     * @example
     * $drawer->line([ 0, 0 ], [ 100, 100 ], new Color(0, 0, 0));
     *
     * @param int[] $start
     * @param int[] $end
     * @param Color $color
     * @return Drawer
     */
    public function line(array $start, array $end, Color $color): self
    {
        $start = $this->shiftCoordinates($start);
        $end = $this->shiftCoordinates($end);
        imageline($this->resource, $start[0], $start[1], $end[0], $end[1], $color->allocate($this->resource));
        return $this;
    }

    /**
     * Draws the given text on the canvas.
     * The last argument is a configuration array which may contain keys using
     * the following constants:
     * - `Drawer::FONT_SIZE`: The font size of the text in percent. Defaults to 7.5.
     * - `Drawer::TEXT_ALIGN`: Alignment of the text. Can be one of the following values:
     *      - `Drawer::TEXT_ALIGN_LEFT`: Coordinates specify the position of the left edge (default).
     *      - `Drawer::TEXT_ALIGN_CENTER`: Coordinates specify the position of the horizontal center.
     *      - `Drawer::TEXT_ALIGN_RIGHT`: Coordinates specify the position of the right edge.
     * - `Drawer::VERTICAL_ALIGN`: Alignment of the text baseline. Can be one of the following values:
     *      - `Drawer::VERTICAL_ALIGN_TOP`: Coordinates specify the position of the top edge.
     *      - `Drawer::VERTICAL_ALIGN_CENTER`: Coordinates specify the position of the vertical center.
     *      - `Drawer::VERTICAL_ALIGN_BOTTOM`: Coordinates specify the position of the baseline (default).
     * - `Drawer::ANGLE`: The angle at which the text is being drawn. Defaults to 0.
     * - `Drawer::TTF_FONT`: The font that is used to draw the text. This must be an absolute
     *                       path to a TTF font file. Defaults to a version of OpenSans that is
     *                       bundled with the library.
     *
     * @example
     * $drawer->text(
     *      'test',
     *      [ 100, 100 ],
     *      new Color(0, 0, 0),
     *      [ Drawer::TEXT_ALIGN => Drawer::TEXT_ALIGN_CENTER ]
     * );
     *
     * @param string $text
     * @param int[] $position
     * @param Color $color
     * @param array $configuration
     * @return Drawer
     */
    public function text(string $text, array $position, Color $color, array $configuration = []): self {
        $configuration = array_merge([
            static::FONT_SIZE => 7.5,
            static::TEXT_ALIGN => static::TEXT_ALIGN_LEFT,
            static::VERTICAL_ALIGN => static::VERTICAL_ALIGN_BOTTOM,
            static::ANGLE => 0,
            static::TTF_FONT => __DIR__ . '/../../resources/font/OpenSans-Regular.ttf',
        ], $configuration);
        $position = $this->shiftCoordinates($position);

        // Convert the percent values to pixels and then to points
        // GD uses an internal DPI of 96.
        $fontSizePixels = ($configuration[static::FONT_SIZE] / 100) * $this->size[1];
        $fontSizePoint = $fontSizePixels * .75;

            // Measure the text
        $textSize = imagettfbbox(
            $fontSizePoint,
            $configuration[static::ANGLE],
            $configuration[static::TTF_FONT],
            $text
        );
        $textWidth = abs($textSize[0] - $textSize[2]);
        $textHeight = abs($textSize[1] - $textSize[7]);

        // Shift position according to alignment
        switch($configuration[static::TEXT_ALIGN]) {
            case static::TEXT_ALIGN_CENTER:
                $position[0] -= $textWidth / 2;
                break;
            case static::TEXT_ALIGN_RIGHT:
                $position[0] -= $textWidth;
                break;
            case static::TEXT_ALIGN_LEFT:
            default:
                // Nothing needs to be changed, text is left aligned by default.
                break;
        }
        switch($configuration[static::VERTICAL_ALIGN]) {
            case static::VERTICAL_ALIGN_CENTER:
                $position[1] += $textHeight / 2;
                break;
            case static::VERTICAL_ALIGN_TOP:
                $position[1] += $textHeight;
                break;
            case static::VERTICAL_ALIGN_BOTTOM:
            default:
                // Nothing needs to be changed, text is bottom aligned by default.
                break;
        }

        // Draw text
        imagettftext(
            $this->resource,
            $fontSizePoint,
            $configuration[static::ANGLE],
            $position[0],
            $position[1],
            $color->allocate($this->resource),
            $configuration[static::TTF_FONT],
            $text
        );

        return $this;
    }

    /**
     * Draws a filled polygon to the canvas.
     * This The first argument must be an array of coordinate arrays
     * where each coordinate represents one point.
     *
     * @example
     * // Draw a triangle with a 90 degree edge
     * $drawer->filledPolygon(
     *      [
     *          [ 0,   0   ],    // First point (0, 0)
     *          [ 100, 100 ],   // Second point (100, 100)
     *          [ 0,   100 ],   // Third point at (0, 100)
     *      ],
     *      new Color(128, 128, 128)
     * );
     *
     * @param array $coordinates
     * @param Color $color
     * @return Drawer
     */
    public function filledPolygon(array $coordinates, Color $color): self
    {
        $points = [];
        foreach ($coordinates as $index => $coordinate) {
            $coordinate = $this->shiftCoordinates($coordinate);
            $points[] = $coordinate[0];
            $points[] = $coordinate[1];
        }

        imagefilledpolygon(
            $this->resource,
            $points,
            floor(count($points) / 2),
            $color->allocate($this->resource)
        );

        return $this;
    }


    /**
     * Draws a stroked polygon to the canvas.
     * This The first argument must be an array of coordinate arrays
     * where each coordinate represents one point.
     *
     * @example
     * // Draw a triangle with a 90 degree edge
     * $drawer->strokedPolygon(
     *      [
     *          [ 0,   0   ],    // First point (0, 0)
     *          [ 100, 100 ],   // Second point (100, 100)
     *          [ 0,   100 ],   // Third point at (0, 100)
     *      ],
     *      new Color(128, 128, 128)
     * );
     *
     * @param array $coordinates
     * @param Color $color
     * @return Drawer
     */
    public function strokedPolygon(array $coordinates, Color $color): self
    {
        $points = [];
        foreach ($coordinates as $index => $coordinate) {
            $coordinate = $this->shiftCoordinates($coordinate);
            $points[] = $coordinate[0];
            $points[] = $coordinate[1];
        }

        imagepolygon(
            $this->resource,
            $points,
            floor(count($points) / 2),
            $color->allocate($this->resource)
        );

        return $this;
    }

    /**
     * Returns a new drawer that has been shifted by the given percentage.
     * @param array $shiftBy
     * @param array|null $size
     * @return Drawer
     */
    public function shiftDrawingCanvas(array $shiftBy, array $size = null): self
    {
        $shiftBy = [
            ($shiftBy[0] / 100) * $this->size[0],
            ($shiftBy[1] / 100) * $this->size[1],
        ];
        $newOffset = [
            $this->offset[0] + $shiftBy[0],
            $this->offset[1] + $shiftBy[1]
        ];
        if ($size) {
            $size = [
                ($size[0] / 100) * $this->size[0],
                ($size[1] / 100) * $this->size[1],
            ];
        } else {
            $size = [ $this->size[0] - $shiftBy[0], $this->size[1] - $shiftBy[1] ];
        }

        return new static($this->resource, $newOffset, $size);
    }

    /**
     * Takes raw coordinate inputs and applies the configured
     * offset to it.
     *
     * @param int[] $coordinates
     * @return int[]
     */
    protected function shiftCoordinates(array $coordinates): array {
        return [
            $this->offset[0] + ($coordinates[0] / 100) * $this->size[0],
            $this->offset[1] + ($coordinates[1] / 100) * $this->size[1],
        ];
    }
}
