<?php declare(strict_types=1);

namespace Newland\Toubiz\Sync\Neos\Tests\Unit\Domain\Repository;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityManagerInterface;
use Neos\Flow\Tests\FunctionalTestCase;
use Newland\Toubiz\Sync\Neos\Domain\Model\RecordConfiguration;
use Newland\Toubiz\Sync\Neos\Domain\Repository\RecordConfigurationRepository;
use Newland\Toubiz\Sync\Neos\Tests\Unit\Domain\Repository\Mock\TestModel;
use Ramsey\Uuid\Uuid;

class RecordConfigurationRepositoryTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;

    /** @var RecordConfigurationRepository */
    protected $subject;

    public function setUp(): void
    {
        parent::setUp();
        $this->subject = $this->objectManager->get(RecordConfigurationRepository::class);
    }

    public function testFindsExistingRecordConfiguration(): void
    {
        $model = new TestModel();
        $model->persistenceObjectIdentifier = (string) Uuid::uuid4();

        $configuration = new RecordConfiguration();
        $configuration->setPersistenceObjectIdentifier($model->getPersistenceObjectIdentifier());
        $configuration->setConfiguration([ 'foo' => 'bar' ]);
        $this->subject->add($configuration);
        $this->persistenceManager->persistAll();

        $fromDb = $this->subject->forRecord($model);
        $this->assertEquals($model->getPersistenceObjectIdentifier(), $fromDb->getPersistenceObjectIdentifier());
        $this->assertEquals('bar', $fromDb->getConfiguration()['foo'] ?? null);
    }

    public function testCreatesEmptyRecordConfigurationIfNonExistent(): void
    {
        $model = new TestModel();
        $model->persistenceObjectIdentifier = (string) Uuid::uuid4();

        $this->assertEmpty($this->subject->forRecord($model)->getConfiguration());
    }

    public function testUsesFindsRecordConfigurationCreatedByUpdatingIt(): void
    {
        $model = new TestModel();
        $model->persistenceObjectIdentifier = (string) Uuid::uuid4();

        $this->subject->updateConfiguration($model, function(array $config) {
            $config['test'] = 'test';
            return $config;
        }, false);
        $this->persistenceManager->persistAll();

        $fromDb = $this->subject->forRecord($model);
        $this->assertEquals($model->getPersistenceObjectIdentifier(), $fromDb->getPersistenceObjectIdentifier());
        $this->assertEquals('test', $fromDb->getConfiguration()['test'] ?? null);
    }

    public function testAppliesRecordConfiguration(): void
    {
        $model = new TestModel();
        $model->persistenceObjectIdentifier = (string) Uuid::uuid4();

        $this->subject->updateConfiguration($model, function(array $config) {
            $config['test'] = 'test';
            return $config;
        }, false);

        $this->assertCount(1, $model->recordConfigurationsApplied);
        $this->assertEquals('test', $model->recordConfigurationsApplied[0]->getConfiguration()['test']);
    }

}
