<?php
namespace Newland\Toubiz\Sync\Neos\Importer;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\Service\GeoLocation\GeoLocationServiceInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\Address;
use Newland\Toubiz\Sync\Neos\Domain\Repository\AddressRepository;
use Newland\Toubiz\Sync\Neos\Service\GeoLocationFactory;

/**
 * Address importer.
 *
 * @Flow\Scope("singleton")
 *
 * @property AddressAdapterInterface $data
 */
class AddressImporter extends AbstractImporter
{
    /**
     * @var array The parent object being imported.
     */
    protected $parentObject;

    /**
     * @Flow\Inject
     * @var AddressRepository
     */
    protected $addressRepository;

    /**
     * @var GeoLocationServiceInterface
     */
    protected $geoLocationService;

    public function injectGeoLocationService(GeoLocationFactory $factory)
    {
        $this->geoLocationService = $factory->get('Newland.Toubiz.Sync.Neos.services.GeoLocation');
    }

    /**
     * @param mixed $parentObject
     * @return void
     */
    public function setParentObject($parentObject)
    {
        $this->parentObject = $parentObject;
    }

    /**
     * Import method.
     *
     * Persist given data by creating new objects or updating existing ones.
     *
     * @param AddressAdapterInterface $data
     * @return Address
     */
    public function import($data)
    {
        $this->data = $data;

        $address = $this->addressRepository->findOneByOriginalId($data->getExternalId());
        $persisted = (bool) $address;

        if (!$address) {
            $address = new Address;
        }

        if (!$address->getOriginalId()) {
            $address->setOriginalId($data->getExternalId());
        }

        $address->setName($data->getTitle());
        $address->setFirstName($data->getFirstName());
        $address->setLastName($data->getLastName());
        $address->setStreet($data->getStreet());
        $address->setZip($this->getZip($address));
        $address->setCity($data->getCity());
        $address->setPhoneNumber($data->getPhoneNumber());
        $address->setEmailAddress($data->getEmailAddress());
        $address->setWebsiteAddress($data->getWebsiteAddress());

        $this->mapCoordinates($address);

        if ($persisted) {
            $this->addressRepository->update($address);
        } else {
            $this->addressRepository->add($address);
        }

        // Persisting is required due to the API returning the same object multiple times!
        $this->persistenceManager->persistAll();

        return $address;
    }

    private function mapCoordinates(Address $address)
    {
        $latitude = (float) $this->data->getLatitude();
        $longitude = (float) $this->data->getLongitude();
        if (abs($latitude) > .1 && abs($longitude) > .1) {
            $address->setLatitude($latitude);
            $address->setLongitude($longitude);
        } else {
            $address->setLatitude(null);
            $address->setLongitude(null);
        }
    }

    private function getZip(Address $address)
    {
        $zip = $this->data->getZip();

        if (empty($address->getZip()) && $this->data->isLookup()) {
            $zip = $this->geoLocationService->getZipFromGeoLocation(
                $this->data->getLatitude(),
                $this->data->getLongitude()
            );
        }

        return $zip;
    }
}
