<?php declare(strict_types=1);

namespace Newland\Toubiz\Poi\Neos\Tests\Unit\Command;

use Neos\Flow\Tests\FunctionalTestCase;
use Newland\Toubiz\Api\Service\ServiceFactory;
use Newland\Toubiz\Api\Service\Tportal;
use Newland\Toubiz\Sync\Neos\Command\Helper\ConfigurationHelper;
use Newland\Toubiz\Sync\Neos\Command\PackagesCommandController;
use PHPUnit\Framework\MockObject\MockObject;

class PackagesCommandControllerTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;

    /** @var PackagesCommandController */
    private $subject;

    /** @var MockObject<Tportal\ApiService> */
    private $tportalApiService;

    public function setUp(): void
    {
        parent::setUp();

        $this->subject = $this->objectManager->get(PackagesCommandController::class);

        $this->tportalApiService = $this->createMock(Tportal\ApiService::class);
        ServiceFactory::mock('Tportal/Api', $this->tportalApiService);
    }

    public function tearDown(): void
    {
        ServiceFactory::resetMocks();
        $this->objectManager->forgetInstance(ConfigurationHelper::class);
        parent::tearDown();
    }

    private function setServiceOptions(array $options): void
    {
        $configurationHelper = new ConfigurationHelper();
        $this->inject($configurationHelper, 'configuration', [ 'services' => $options ]);
        $this->inject($this->subject, 'configurationHelper', $configurationHelper);
    }

    public function testCallsCorrectMethodOnApiServiceToImportOffers(): void
    {
        $this->setServiceOptions([
             'Tportal/Api' => [
                 'clients' => [
                     'foo' => [ 'client' => 'foo' ]
                 ]
             ]
         ]);

        $this->tportalApiService->expects($this->once())
            ->method('fetchPackages');

        $this->subject->synchronizeCommand(true, false);
    }

    public function testPassesDownClientWhenImportingLodgings(): void
    {
        $this->setServiceOptions([
             'Tportal/Api' => [
                 'clients' => [
                     'foo' => [ 'client' => 'foo' ]
                 ]
             ]
         ]);

        $this->tportalApiService->method('fetchPackages')->willReturn(null);
        $this->tportalApiService->expects($this->once())
            ->method('setClientName')
            ->with($this->equalTo('foo'));

        $this->subject->synchronizeCommand(true, false);
    }

    public function testPassesDownLanguagesWhenImportingLodgings(): void
    {
        $this->setServiceOptions([
             'Tportal/Api' => [
                 'clients' => [
                     'foo' => [
                         'client' => 'foo',
                         'languages' => [ 'en', 'fr', 'es' ]
                     ]
                 ]
             ]
         ]);

        $this->tportalApiService->method('fetchPackages')->willReturn(null);
        $this->tportalApiService->expects($this->exactly(3))
            ->method('setLanguage')
            ->withConsecutive(
                [ $this->equalTo('en') ],
                [ $this->equalTo('fr') ],
                [ $this->equalTo('es') ]
            );

        $this->subject->synchronizeCommand(true, false);
    }

    public function testDefaultsToGerman(): void
    {
        $this->setServiceOptions([
             'Tportal/Api' => [
                 'clients' => [
                     'foo' => [ 'client' => 'foo' ]
                 ]
             ]
         ]);

        $this->tportalApiService->method('fetchPackages')->willReturn(null);
        $this->tportalApiService->expects($this->once())
            ->method('setLanguage')
            ->with($this->equalTo('de'));

        $this->subject->synchronizeCommand(true, false);
    }

}
