<?php
namespace Newland\Toubiz\Sync\Neos\Importer;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\Constants\EntityListType;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleTypeCityAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\Service\WithUuidPredictionService;
use Newland\Toubiz\Api\Service\WithUuidPredictionServiceInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Domain\Model\CityData;
use Newland\Toubiz\Sync\Neos\Domain\Model\Medium;
use Newland\Toubiz\Sync\Neos\Domain\Model\RelatedLists\EntityList;
use Newland\Toubiz\Sync\Neos\Domain\Model\RelatedLists\EntityListInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\RelatedLists\RelatedLists;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;

/**
 * Article importer.
 *
 * @Flow\Scope("singleton")
 */
class ArticleImporter extends AbstractImporter
{
    use WithUuidPredictionService;

    /**
     * @var Article
     */
    protected $article;

    /**
     * @Flow\Inject
     * @var ArticleRepository
     */
    protected $articleRepository;

    /**
     * @var string
     */
    protected $client = '';

    /**
     * @param ArticleAdapterInterface $data
     * @return void
     */
    public function remove(ArticleAdapterInterface $data): void
    {
        $this->initialize($data);

        if ($this->article !== null) {
            $this->articleRepository->remove($this->article);
            $this->persistenceManager->persistAll();
        }
    }

    /**
     * Import method.
     *
     * Persist given data by creating new objects or updating existing ones.
     *
     * @param ArticleAdapterInterface $data
     * @return void
     */
    public function import($data): void
    {
        $this->initialize($data);
        $persisted = $this->article !== null;

        if ($this->article === null) {
            $this->article = new Article();
        }

        $this->mapSimpleValues($data);
        $this->mapRatingValues($data);
        $this->article->setClient($this->client);
        $this->mapOpeningTimes($data);

        $this->mapGeoCoordinates($data);
        $this->mapAddresses($data);
        $this->mapCategories($data);
        $this->mapFiles($data);
        $this->mapMedia($data);
        $this->mapAttributes($data);

        if ($data instanceof ArticleTypeCityAdapterInterface) {
            $this->mapCityData($data);
        }

        $this->article->setUpdatedAt(new \DateTime);

        if ($persisted) {
            $this->articleRepository->update($this->article);
        } else {
            $this->articleRepository->add($this->article);
        }
    }

    private function initialize(ArticleAdapterInterface $data): void
    {
        $language = $data->getLanguage();
        if ($language) {
            $this->articleRepository->setLanguage($language);
        }

        if ($this->client) {
            $article = $this->articleRepository->findOneByOriginalIdAndClient(
                $data->getExternalId(),
                $this->client
            );
        } else {
            $article = $this->articleRepository->findOneByOriginalId($data->getExternalId());
        }

        if ($article !== null) {
            $this->article = $article;
        }
    }

    public function setClient(string $client): void
    {
        $this->client = $client;
    }

    protected function mapSimpleValues(ArticleAdapterInterface $data): void
    {
        $this->article->setOriginalId($data->getExternalId());
        $this->article->setMainType($data->getMainType());
        $this->article->setName($data->getName());
        $this->article->setAbstract($data->getAbstract());
        $this->article->setDescription($data->getDescription());
        $this->article->setFacebookUri($data->getFacebookUri());
        $this->article->setTwitterUri($data->getTwitterUri());
        $this->article->setInstagramUri($data->getInstagramUri());
        $this->article->setYoutubeUri($data->getYoutubeUri());
        $this->article->setWikipediaUri($data->getWikipediaUri());
        $this->article->setFlickrUri($data->getFlickrUri());
        $this->article->setSourceName($data->getSourceName());
        $this->article->setAuthorName($data->getAuthorName());
        $this->article->setBookingUri($data->getBookingUri());
        $this->article->setDetailUri($data->getDetailUri());
        $this->article->setLanguage($data->getLanguage());
        $this->article->setUrlIdentifier($this->generateUrlIdentifierFromData($data));

        $mainAddressAdapter = $data->getMainAddress();
        if ($mainAddressAdapter) {
            $this->article->setMainAddress(
                (new AddressImporter())->import($mainAddressAdapter)
            );
        } else {
            $this->article->setMainAddress(null);
        }
    }

    protected function mapRatingValues(ArticleAdapterInterface $data): void
    {
        $this->article->setStarClassification($data->getStarClassification() ?? 0);
        $this->article->setAverageRating($data->getAverageRating() ?? 0);
        $this->article->setNumberOfRatings($data->getNumberOfRatings() ?? 0);
    }

    protected function mapGeoCoordinates(ArticleAdapterInterface $data): void
    {
        $this->article->setLatitude($data->getLatitude());
        $this->article->setLongitude($data->getLongitude());
    }

    protected function mapAddresses(ArticleAdapterInterface $data): void
    {
        $addresses = $this->article->getAddresses();
        $addresses->clear();

        foreach ($data->getAddresses() as $address) {
            $entity = (new AddressImporter())->import($address);
            if (!$addresses->contains($entity)) {
                $addresses->add($entity);
            }
        }

        $this->article->setAddresses($addresses);
    }

    protected function mapCategories(ArticleAdapterInterface $data): void
    {
        $categories = $this->article->getCategories();
        $categories->clear();

        $importer = new CategoryImporter();
        $importer->setLanguage($data->getLanguage());

        foreach ($data->getCategories() as $entry) {
            $category = $importer->import($entry);

            if (!$categories->contains($category)) {
                $categories->add($category);
            }
        }

        $this->article->setCategories($categories);

        $sorting = array_map(
            function (Category $category) {
                return $category->getOriginalId();
            },
            $categories->toArray()
        );

        $this->article->setRelationSortingForField('categories', $sorting);
    }

    protected function mapMedia(ArticleAdapterInterface $data): void
    {
        $media = $this->article->getMedia();
        $media->clear();

        foreach ($data->getMedia() as $entry) {
            $medium = (new MediumImporter)->import($entry);
            if (!$media->contains($medium)) {
                $media->add($medium);
            }
        }
        $this->article->setMedia($media);

        $sorting = array_map(
            function (Medium $medium) {
                return $medium->getOriginalId();
            },
            $media->toArray()
        );

        $this->article->setRelationSortingForField('media', $sorting);
    }

    protected function mapFiles(ArticleAdapterInterface $data): void
    {
        $files = $this->article->getFiles();
        $files->clear();

        foreach ($data->getFiles() as $entry) {
            $file = (new FileImporter)->import($entry);
            if (!$files->contains($file)) {
                $files->add($file);
            }
        }
        $this->article->setFiles($files);
    }

    public function mapOpeningTimes(ArticleAdapterInterface $data): void
    {
        if ($data->getOpeningTimesFormat()) {
            $this->article->setOpeningTimesFormat((string) $data->getOpeningTimesFormat());
            $this->article->setOpeningTimes($data->getOpeningTimes() ?? '');
        }
    }

    public function deleteOldRecords(\DateInterval $timeToLive, int $mainType = null): int
    {
        $i = 0;
        $this->articleRepository->withoutLanguageHandling(
            function () use (&$i, $timeToLive, $mainType) {
                $i = $this->articleRepository->deleteOldRecords($timeToLive, $mainType);
            }
        );
        return $i;
    }

    protected function mapAttributes(ArticleAdapterInterface $data): void
    {
        $attributes = $this->article->getAttributes();
        $attributes->clear();

        $attributesData = $data->getAttributes();
        if ($attributesData) {
            foreach ($attributesData as $attributeData) {
                $importer = new AttributeImporter();
                $importer->setArticle($this->article);
                $attribute = $importer->import($attributeData);
                if (!$attributes->contains($attribute)) {
                    $attributes->add($attribute);
                }
            }
        }

        $this->article->setAttributes($attributes);
    }

    protected function mapRelatedLists(ArticleTypeCityAdapterInterface $data): void
    {
        if ($data instanceof WithUuidPredictionServiceInterface) {
            $data->setUuidPredictionService($this->uuidPredictionService);
        }

        $relatedListsData = $data->getRelatedLists();

        if ($relatedListsData === null) {
            return;
        }

        $relatedLists = new RelatedLists();
        $poiHighlightsData = $relatedListsData->getHighlightsList();
        if ($poiHighlightsData) {
            $entityList = new EntityList();
            $entityList->setType(EntityListType::ARTICLES);
            $entityList->setTitle($poiHighlightsData->getTitle());
            $entityList->setIdentifiers($poiHighlightsData->getIdentifiers());

            $relatedLists->setPoiHighlights($entityList);
        }

        $entityLists = [];
        foreach ($relatedListsData->getEntityLists() as $entityListData) {
            $entityList = new EntityList();
            $entityList->setType($entityListData->getType());
            $entityList->setTitle($entityListData->getTitle());
            $entityList->setIdentifiers($entityListData->getIdentifiers());

            $entityLists[] = $entityList;
        }

        $relatedLists->setEntityLists($entityLists);

        $this->article->setRelatedLists($relatedLists);
    }

    private function mapCityData(ArticleTypeCityAdapterInterface $data): void
    {
        $this->mapRelatedLists($data);

        $cityData = $this->article->getCityData();
        if ($cityData === null) {
            $cityData = new CityData();
        }

        $cityData->setCity($this->article);
        $cityData->setIdToubiz($data->getIdToubiz());
        $cityData->setIdTomas($data->getIdTomas());
        $cityData->setZipCodes($data->getZipCodes());
        $cityData->setClaim($data->getClaim());
        $cityData->setFacts($data->getFacts());

        $this->article->setCityData($cityData);
    }

    private function generateUrlIdentifierFromData(ArticleAdapterInterface $data): string
    {
        return $data->getExternalId() . '-'
            . $this->client . '-'
            . $data->getMainType() . '-'
            . $data->getLanguage();
    }
}
