<?php declare(strict_types=1);

namespace Newland\Toubiz\Sync\Neos\Command;

use Doctrine\ORM\Mapping\ClassMetadata;
use Neos\Flow\Annotations as Flow;
use Doctrine\ORM\EntityManagerInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Model\EventDate;
use Newland\Toubiz\Sync\Neos\Utility\DatabaseUtility;
use Symfony\Component\Console\Output\NullOutput;

class IdentifierCommandController extends AbstractCommandController
{

    protected $types = [
        'article' => Article::class,
        'category' => Category::class,
        'event' => Event::class,
        'eventDate' => EventDate::class,
    ];

    /**
     * @var EntityManagerInterface
     * @Flow\Inject()
     */
    protected $entityManager;

    /**
     * Regenerates the persistence object identifiers of entities that use deterministic
     * identifiers. Identifiers that are not yet deterministic will be changed and references
     * to the old identifier will be updated.
     *
     * @param string|null $type Entity type to change. Can be one of ['article', 'category', 'event', 'eventDate'].
     *     Defaults to all of them.
     * @param bool $quiet Do not print status information to the console. Defaults to `false`.
     */
    public function regenerateCommand(string $type = null, bool $quiet = false): void
    {
        if ($quiet) {
            $this->output->setOutput(new NullOutput());
        }

        if ($type === null) {
            foreach ($this->types as $className) {
                $this->regenerateIdentifiersForClass($className);
            }
        } else {
            $this->regenerateIdentifiersForClass($this->types[$type]);
        }
    }

    private function regenerateIdentifiersForClass(string $className)
    {
        $this->output->outputLine("\n\n" . $className);
        $query = $this->persistenceManager->createQueryForType($className);
        $this->output->progressStart($query->count());

        $updated = 0;
        foreach ($query->execute() as $item) {
            $before = $item->getPersistenceObjectIdentifier();
            $after = $item->generateUuid()->toString();

            if ($before !== $after) {
                $item->setPersistenceObjectIdentifier($after);
                $updated++;

                DatabaseUtility::withoutForeignKeyChecks(
                    $this->entityManager,
                    function () use ($className, $item, $before, $after) {
                        $this->updateAllRelations($className, $before, $after);
                        $this->persistenceManager->update($item);
                        $this->persistenceManager->persistAll();
                    }
                );

                $this->updateNodeContents($before, $after);
            }

            $this->output->progressAdvance();
            $this->output(sprintf(' new identifiers: %d', $updated));
        }

        $this->output->progressFinish();
        $this->output(sprintf(' new identifiers: %d', $updated));
    }

    private function updateAllRelations($className, $before, $after): void
    {
        /** @var ClassMetadata $metadata */
        $metadata = $this->entityManager->getMetadataFactory()->getMetadataFor($className);
        foreach ($metadata->getAssociationMappings() as $associationName => $configuration) {
            [ $tableName, $columnName ] =
                $this->extractTableAndColumnFromAssociationMapping($metadata, $configuration);

            if ($tableName && $columnName) {
                $sql = sprintf(
                    'UPDATE %s SET %s="%s" WHERE %s="%s"',
                    $tableName,
                    $columnName,
                    $after,
                    $columnName,
                    $before
                );
                $this->entityManager->getConnection()->exec($sql);
            }
        }
    }

    private function updateNodeContents(string $before, string $after): void
    {
        $sql = sprintf(
            'UPDATE %s SET properties=REPLACE(properties, "%s", "%s") WHERE properties LIKE "%s"',
            'neos_contentrepository_domain_model_nodedata',
            $before,
            $after,
            '%' . $before . '%'
        );

        $this->entityManager->getConnection()->exec($sql);
    }


    private function extractTableAndColumnFromAssociationMapping(ClassMetadata $metadata, array $configuration)
    {
        $joinTable = $configuration['joinTable'] ?? null;
        $targetEntity = $configuration['targetEntity'] ?? null;
        $mappedBy = $configuration['mappedBy'] ?? null;
        $joinColumns = $configuration['joinColumns'] ?? null;

        if ($joinTable) {
            return [
                $joinTable['name'] ?? null,
                $joinTable['joinColumns'][0]['name'] ?? null
            ];
        }

        if ($joinColumns) {
            return [
                $metadata->getTableName(),
                $joinColumns[0]['name'] ?? null
            ];
        }

        if ($targetEntity && $mappedBy) {
            /** @var ClassMetadata $foreignMetadata */
            $foreignMetadata = $this->entityManager->getMetadataFactory()->getMetadataFor($targetEntity);
            $foreignConfiguration = $foreignMetadata->getAssociationMapping($mappedBy);

            return [
                $foreignMetadata->getTableName(),
                $foreignConfiguration['joinColumns'][0]['name'] ?? null
            ];
        }

        return [ null, null ];
    }
}
