<?php
namespace Newland\Toubiz\Sync\Neos\Tests\Unit\Domain\Repository;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\Common\Collections\ArrayCollection;
use Neos\Flow\Tests\FunctionalTestCase;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Sync\Neos\Domain\Filter\ArticleFilter;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;
use Newland\Toubiz\Sync\Neos\Tests\Factory\ArticleFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\CategoryFactory;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;

class ArticleRepositoryTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;

    /**
     * @var ArticleRepository
     */
    protected $subject;

    public function setUp(): void
    {
        parent::setUp();
        $categoryFactory = new CategoryFactory($this->objectManager);

        $categoryOne = $categoryFactory->create();
        $categoryTwo = $categoryFactory->create();
        $categoryThree = $categoryFactory->create();

        $articleFactory = new ArticleFactory($this->objectManager);
        $articleFactory->create();
        $articleFactory->create(
            [
                'categories' => new ArrayCollection(
                    [
                        $categoryOne,
                        $categoryTwo,
                        $categoryThree,
                    ]
                ),
            ]
        );

        $this->subject = $this->objectManager->get(ArticleRepository::class);
        $this->subject->setStrictLanguageHandling(false);
    }

    public function testFilterFindsArticlesWithNoCategory(): void
    {
        $result = $this->subject->countByFilter(new ArticleFilter(), 10);
        $this->assertEquals(2, $result['items']);
    }

    public function testDataSourceListsOneOptionForEveryCategoryAndOneIfNoCategoryAttached(): void
    {
        $this->assertCount(4, $this->subject->findAllForDataSource(new ArticleFilter()));
    }

    /**
     * @dataProvider provideArticleTypes
     * @param int $type
     */
    public function testEmptyFilterFindsArticlesOfAllTypes(int $type): void
    {
        $article = (new ArticleFactory($this->objectManager))->create([ 'mainType' => $type ]);
        $ids = array_map(
            function(Article $article) { return $article->getPersistenceObjectIdentifier(); },
            $this->subject->findByFilter(new ArticleFilter())
        );
        $this->assertContains($article->getPersistenceObjectIdentifier(), $ids);
    }


    /**
     * @dataProvider provideArticleTypes
     * @param int $type
     */
    public function testFilterForArticleTypeFindsArticlesOfThatType(int $type): void
    {
        $article = (new ArticleFactory($this->objectManager))->create([ 'mainType' => $type ]);

        $filter = new ArticleFilter();
        $filter->setMainType($type);
        $ids = array_map(
            function(Article $article) { return $article->getPersistenceObjectIdentifier(); },
            $this->subject->findByFilter($filter)
        );
        $this->assertContains($article->getPersistenceObjectIdentifier(), $ids);
    }

    public function provideArticleTypes(): array
    {
        return array_map(
            function(int $type) { return [ $type ]; },
            ArticleConstants::ALL_TYPES
        );
    }
}
