<?php declare(strict_types=1);


namespace Newland\Toubiz\Sync\Neos\Tests\Unit\Domain\Model;


use Neos\Flow\Tests\FunctionalTestCase;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Tests\Factory\EventFactory;

class EventTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;

    /**
     * @dataProvider provideDeterministicIds
     * @param string $expectedId
     * @param array $properties
     */
    public function testDeterministicIdsAreGeneratedCorrectly(string $expectedId, array $properties): void
    {
        $this->assertEquals(
            $expectedId,
            (new EventFactory($this->objectManager))->make($properties)->generateUuid()->toString(),
            'persistence object identifier should be stable: If this test fails then the'
            . ' way ids are being generated has probably changed and old identifiers need to be migrated'
            . ' to this new style.'
        );
    }

    /**
     * @dataProvider provideDeterministicIds
     * @param string $expectedId
     * @param array $properties
     */
    public function testDeterministicIdsAreAutomaticallyGeneratedWhenStoringInDatabase(string $expectedId, array $properties): void
    {
        /** @var Event $event */
        $event = (new EventFactory($this->objectManager))->create($properties);

        $this->assertEquals(
            $expectedId,
            $event->getPersistenceObjectIdentifier(),
            'persistence object identifier should be stable: If this test fails then the'
            . ' way ids are being generated has probably changed and old identifiers need to be migrated'
            . ' to this new style.'
        );

        // Entity is explicitly deleted to avoid id collisions.
        $this->persistenceManager->remove($event);
        $this->persistenceManager->persistAll();
    }

    public function provideDeterministicIds(): array
    {
        return [
            'baseEvent' => [
                'ebfbcbc9-9eda-543b-b716-82233272dc08',
                [
                    'language' => 'de',
                    'originalId' => 'lalala',
                    'title' => 'First name'
                ]
            ],
            'sameAsBaseButDifferentName: Should have same identifier' => [
                'ebfbcbc9-9eda-543b-b716-82233272dc08',
                [
                    'language' => 'de',
                    'originalId' => 'lalala',
                    'title' => 'Same basic event as before but with different name.'
                ]
            ],
            'languageChanged' => [
                'ebfbcbc9-9eda-543b-b716-82233272dc08',
                [
                    'language' => 'en',
                    'originalId' => 'lalala',
                ]
            ],
            'originalIdChanged' => [
                '5c317d77-3e4b-558d-bde2-a805be1bf596',
                [
                    'language' => 'de',
                    'originalId' => 'blablabla',
                ]
            ],
            'differentPropertiesAllTogether' => [
                '6caa72f8-af9a-5602-9516-50f739a7e3a6',
                [
                    'language' => 'fr',
                    'originalId' => 'foooooo',
                ]
            ],
        ];
    }

}