<?php declare(strict_types=1);

namespace Newland\Toubiz\Poi\Neos\Tests\Unit\Command;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\EntityManagerInterface;
use Neos\Flow\Tests\FunctionalTestCase;
use Newland\Toubiz\Sync\Neos\Command\CategoryCommandController;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Model\RelatedLists\EntityList;
use Newland\Toubiz\Sync\Neos\Domain\Model\RelatedLists\RelatedLists;
use Newland\Toubiz\Sync\Neos\Domain\Repository\CategoryRepository;
use Newland\Toubiz\Sync\Neos\Tests\Factory\ArticleFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\CategoryFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\EventFactory;

class CategoryCommandControllerTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;

    /** @var CategoryCommandController */
    private $subject;

    /** @var CategoryRepository */
    private $categoryRepository;

    /** @var CategoryFactory */
    private $categoryFactory;

    /** @var ArticleFactory */
    private $articleFactory;

    /** @var EventFactory */
    private $eventFactory;

    public function setUp(): void
    {
        parent::setUp();
        $this->subject = $this->objectManager->get(CategoryCommandController::class);
        $this->categoryRepository = $this->objectManager->get(CategoryRepository::class);
        $this->articleFactory = new ArticleFactory($this->objectManager);
        $this->categoryFactory = new CategoryFactory($this->objectManager);
        $this->eventFactory = new EventFactory($this->objectManager);
    }

    public function testDeletesCategoriesNotRelatedToArticlesOrEvents()
    {
        [ $onlyRelatedToArticles, $onlyRelatedToEvents, $relatedToBoth, $relatedToNone ] = $this->categoryFactory->createMultiple(4);

        /** @var Article $article */
        $article = $this->articleFactory->create();
        $article->setCategories(new ArrayCollection([ $onlyRelatedToArticles, $relatedToBoth ]));
        $this->persistenceManager->update($article);

        /** @var Event $event */
        $event = $this->eventFactory->create();
        $event->setCategories(new ArrayCollection([ $onlyRelatedToEvents, $relatedToBoth ]));
        $this->flushToDatabaseAndClearCaches([ $event ]);

        $this->callRemoveUnusedCommand();

        $this->assertNotNull($this->categoryRepository->findByIdentifier($onlyRelatedToArticles->getPersistenceObjectIdentifier()), 'Category with relations to article should not be deleted');
        $this->assertNotNull($this->categoryRepository->findByIdentifier($onlyRelatedToEvents->getPersistenceObjectIdentifier()), 'Category with relations to event should not be deleted');
        $this->assertNotNull($this->categoryRepository->findByIdentifier($relatedToBoth->getPersistenceObjectIdentifier()), 'Category with relations to article & events should not be deleted');
        $this->assertNull($this->categoryRepository->findByIdentifier($relatedToNone->getPersistenceObjectIdentifier()), 'Category with relations to neither articles nor events should be deleted');
    }

    public function testDeletesCategoriesOfAllLanguages(): void
    {
        [ $usedGerman, $unusedGerman ] = $this->categoryFactory->createMultiple(2, [ 'language' => 'de' ]);
        [ $usedEnglish, $unusedEnglish ] = $this->categoryFactory->createMultiple(2, [ 'language' => 'en' ]);

        $this->articleFactory->create([ 'language' => 'de', 'categories' => new ArrayCollection([ $usedGerman ]) ]);
        $this->eventFactory->create([ 'language' => 'en', 'categories' => new ArrayCollection([ $usedEnglish ]) ]);
        $this->flushToDatabaseAndClearCaches();

        $this->callRemoveUnusedCommand();

        $this->categoryRepository->setLanguage('de');
        $this->assertNotNull($this->categoryRepository->findByIdentifier($usedGerman->getPersistenceObjectIdentifier()), 'Used german category should not be deleted.');
        $this->assertNull($this->categoryRepository->findByIdentifier($unusedGerman->getPersistenceObjectIdentifier()), 'Unused german category should be deleted.');

        $this->categoryRepository->setLanguage('en');
        $this->assertNotNull($this->categoryRepository->findByIdentifier($usedEnglish->getPersistenceObjectIdentifier()), 'Used english category should not be deleted.');
        $this->assertNull($this->categoryRepository->findByIdentifier($unusedEnglish->getPersistenceObjectIdentifier()), 'Unused english category should be deleted');
    }

    public function testRespectsSoftRelationsOfArticleLists()
    {
        [ $used, $unused ] = $this->categoryFactory->createMultiple(2);
        $usedIdentifier = $used->getPersistenceObjectIdentifier();
        $unusedIdentifier = $unused->getPersistenceObjectIdentifier();

        $related = new RelatedLists();
        $related->addEntityList(new EntityList(Category::class, [ $used->getOriginalId() ]));
        $article = $this->articleFactory->create([ 'relatedLists' => $related ]);

        /** @var EntityManagerInterface $entityManager */
        $entityManager = $this->objectManager->get(EntityManagerInterface::class);
        $result = $entityManager->getConnection()->executeQuery(
            'SELECT relatedlists 
            FROM newland_toubiz_sync_neos_domain_model_article
             WHERE persistence_object_identifier="' . $article->getPersistenceObjectIdentifier() . '"'
        );

        $this->flushToDatabaseAndClearCaches();

        $this->callRemoveUnusedCommand();

        $this->assertNotNull($this->categoryRepository->findByIdentifier($usedIdentifier), 'Category with soft relations should not be deleted');
        $this->assertNull($this->categoryRepository->findByIdentifier($unusedIdentifier), 'Category without soft relations should be deleted');
    }

    private function callRemoveUnusedCommand(): void
    {
        $this->subject->removeUnusedCommand(true);
        $this->flushToDatabaseAndClearCaches();
    }

    private function flushToDatabaseAndClearCaches($entitiesToUpdate = []): void
    {
        foreach ($entitiesToUpdate as $entity) {
            $this->persistenceManager->update($entity);
        }
        $this->persistenceManager->persistAll();
        $this->persistenceManager->clearState();
    }
}