<?php
namespace Newland\Toubiz\Sync\Neos\Importer;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\ObjectAdapter\EventAdapterInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventRepository;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventDateRepository;

/**
 * Event importer.
 *
 * @Flow\Scope("singleton")
 */
class EventImporter extends AbstractImporter
{
    /**
     * @var EventAdapterInterface
     */
    protected $data;

    /**
     * @var Event
     */
    protected $event;

    /**
     * @Flow\Inject()
     * @var EventRepository
     */
    protected $eventRepository;

    /**
     * @Flow\Inject()
     * @var EventDateRepository
     */
    protected $eventDateRepository;

    /**
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Events.Neos", path="expireDays")
     * @var int
     */
    protected $expireDays;

    /**
     * Post-import.
     *
     * Should be executed after importing multiple records.
     *
     * @return void
     */
    public function postImport()
    {
        $endsAt = new \DateTime();
        $endsAt->setTime(0, 0, 0);

        if ($this->expireDays === null) {
            $this->expireDays = 30;
        }

        $endsAt->modify('-' . $this->expireDays . ' days');

        $this->eventDateRepository->deletePastRecords($endsAt);
    }

    /**
     * Import method.
     *
     * Persist given data by creating new objects or updating existing ones.
     *
     * @param EventAdapterInterface $data
     * @return Event
     */
    public function import($data): Event
    {
        $this->initialize($data);

        $persisted = $this->fetchEventOrCreateNew();

        // Map simple values to have a valid event.
        $this->mapSimpleValues();

        // The event is being persisted in order to create relations to it.
        if ($persisted) {
            $this->eventRepository->update($this->event);
        } else {
            $this->eventRepository->add($this->event);
        }

        $this->mapAddresses();
        $this->mapCategories();
        $this->mapEventDates();
        $this->mapEventTags();
        $this->mapMedia();

        $this->eventRepository->update($this->event);

        // Persisting is required due to the API returning the same object multiple times!
        $this->persistenceManager->persistAll();

        return $this->event;
    }

    protected function mapSimpleValues(): void
    {
        $this->event->setLanguage($this->data->getLanguage());

        $this->event->setOriginalId($this->data->getExternalId());
        $this->event->setTitle($this->data->getName());
        $this->event->setDescription($this->data->getDescription());
        $this->event->setAdditionalInformation($this->data->getAdditionalInformation());
        $this->event->setAdmission($this->data->getAdmission());
        $this->event->setTicketContact($this->data->getTicketContact());
        $this->event->setIsHighlight($this->data->getIsHighlight());
        $this->event->setBeginsAt($this->data->getBeginsAt());
        $this->event->setEndsAt($this->data->getEndsAt());
        $this->event->setUpdatedAt($this->data->getUpdatedAt());
        $this->event->setTicketUri($this->data->getTicketUri() ?? '');
        $this->event->setLink($this->data->getLink() ?? '');
        $this->event->setSourceName($this->data->getSourceName());
        $this->event->setAttributes($this->data->getAttributes());
        $this->event->setScope($this->data->getScope());
        $this->event->setUrlIdentifier($this->generateUrlIdentifierFromData());
    }

    /**
     * Imports and maps addresses.
     *
     * @return void
     */
    protected function mapAddresses()
    {
        $organizerItem = $this->data->getOrganizer();
        $organizer = null;
        if ($organizerItem) {
            $importer = new AddressImporter();
            $organizer = $importer->import($organizerItem);
        }
        $this->event->setOrganizer($organizer);

        $locationItem = $this->data->getLocation();
        $location = null;
        if ($locationItem) {
            $location = (new AddressImporter())->import($locationItem, $this->event->getLocation());
        }
        $this->event->setLocation($location);
    }

    /**
     * Maps categories onto the event.
     *
     * @return void
     */
    protected function mapCategories()
    {
        $categories = $this->event->getCategories();
        $categories->clear();
        foreach ($this->data->getCategories() as $data) {
            $importer = new CategoryImporter();
            $importer->setLanguage($this->data->getLanguage());
            $categories->add($importer->import($data));
        }
        $this->event->setCategories($categories);
    }

    /**
     * Maps event dates onto the event.
     *
     * @todo removal procedure is generic, refactor
     * @return void
     */
    public function mapEventDates()
    {
        $existing = [];
        foreach ($this->event->getEventDates() as $date) {
            $existing[$date->getOriginalId()] = $date;
        }

        $dates = $this->event->getEventDates();
        $dates->clear();

        foreach ($this->data->getEventDates() as $eventDate) {
            $importer = new EventDateImporter();
            $importer->setLanguage($this->data->getLanguage());
            $importer->setEvent($this->event);
            $dates->add(
                $importer->import($eventDate, $existing[$eventDate->getExternalId()] ?? null)
            );
        }

        $this->event->setEventDates($dates);
    }

    /**
     * Maps event tags onto the event.
     *
     * @return void
     */
    public function mapEventTags()
    {
        $existing = [];
        foreach ($this->event->getEventTags() as $tag) {
            $existing[$tag->getOriginalId()] = $tag;
        }
        $eventTags = $this->event->getEventTags();
        $eventTags->clear();

        foreach ($this->data->getEventTags() as $tag) {
            $importer = new EventTagImporter;
            $importer->setLanguage($this->data->getLanguage());
            $eventTags->add(
                $importer->import($tag, $existing[$tag->getExternalId()] ?? null)
            );
        }

        $this->event->setEventTags($eventTags);
    }

    /**
     * Maps multimedia events onto the event.
     *
     * @return void
     */
    public function mapMedia()
    {
        $media = $this->event->getMedia();
        $media->clear();
        foreach ($this->data->getMedia() as $data) {
            $importer = new MediumImporter;
            $importer->setEvent($this->event);
            $media->add($importer->import($data));
        }
        $this->event->setMedia($media);
    }

    private function fetchEventOrCreateNew(): bool
    {
        $event = $this->eventRepository->findOneByOriginalId($this->data->getExternalId());
        $persisted = (bool) $event;
        $this->event = $event ?? new Event();
        return $persisted;
    }

    private function initialize($data): void
    {
        $this->data = $data;
        $this->eventRepository->setLanguage($this->data->getLanguage());
        $this->eventDateRepository->setLanguage($this->data->getLanguage());
    }

    private function generateUrlIdentifierFromData(): string
    {
        return $this->event->getOriginalId() . '-' . $this->event->getLanguage();
    }
}
