<?php declare(strict_types=1);
namespace Newland\Toubiz\Sync\Neos\Domain\Model;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\Mapping as ORM;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Sync\Neos\Domain\Model\Traits\DeterministicPersistenceObjectIdentifier;
use Newland\Toubiz\Sync\Neos\Orm\Uuid\CustomUuidGeneration;
use Newland\Toubiz\Sync\Neos\Orm\Uuid\UuidGenerator;
use Newland\Toubiz\Sync\Neos\Translation\TranslatableEntity;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;

/**
 * A date/time an event is going on.
 *
 * @Flow\Entity
 * @ORM\Table(indexes={
 *      @ORM\Index(name="newland_toubiz_sync_neos_eventdate_import_ident", columns={"language", "originalid"})
 * })
 */
class EventDate extends AbstractEntity implements CustomUuidGeneration
{
    use TranslatableEntity;

    public function generateUuid(): UuidInterface
    {
        return UuidGenerator::uuidFromProperties([ $this->originalId, $this->language ]);
    }

    /**
     * @ORM\ManyToOne(inversedBy="eventDates")
     * @var Event|null
     */
    protected $event;

    /** @var \DateTime */
    protected $beginsAt;

    /** @var \DateTime */
    protected $endsAt;

    /** @var string */
    protected $note;

    /**
     * @ORM\Column(nullable=true)
     * @var string|null Specific detail URI for this event.
     */
    protected $detailUri;

    public function setEvent(?Event $event): void
    {
        $this->event = $event;
    }

    public function getEvent(): ?Event
    {
        return $this->event;
    }

    public function setBeginsAt(\DateTime $beginsAt): void
    {
        $this->beginsAt = $beginsAt;
    }

    public function getBeginsAt(): \DateTime
    {
        return $this->beginsAt;
    }

    public function getBeginsAtSpecificTime(): bool
    {
        return ((int) $this->beginsAt->format('H') > 0);
    }

    public function setEndsAt(\DateTime $endsAt): void
    {
        $this->endsAt = $endsAt;
    }

    public function getEndsAt(): \DateTime
    {
        return $this->endsAt;
    }

    public function getEndsOnSameDay(): bool
    {
        return ($this->beginsAt->format('Y-m-d') === $this->endsAt->format('Y-m-d'));
    }

    public function getIsInThePast(): bool
    {
        $today = new \DateTime;
        return ($this->endsAt->format('Y-m-d H:i') <= $today->format('Y-m-d H:i'));
    }

    public function setNote($note): void
    {
        $this->note = $note;
    }

    public function getNote(): string
    {
        return $this->note;
    }

    public function setDetailUri(?string $detailUri): void
    {
        $this->detailUri = $detailUri;
    }

    public function getDetailUri(): ?string
    {
        return $this->detailUri;
    }
}
