<?php
namespace Newland\Toubiz\Sync\Neos\Command;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\AbstractQuery;
use Newland\Toubiz\Api\ObjectAdapter\OfferAdapterInterface;
use Newland\Toubiz\Api\Service\ServiceFactory;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\Service\Tportal\ApiService;
use Newland\Toubiz\Sync\Neos\Domain\Model\Offer;
use Newland\Toubiz\Sync\Neos\Domain\Repository\OfferRepository;
use Newland\Toubiz\Sync\Neos\Importer\OfferImporter;

/**
 * @Flow\Scope("singleton")
 */
class OfferCommandController extends AbstractCommandController
{
    const TYPE_TPORTAL = 'tportal';

    /**
     * @var OfferRepository
     * @Flow\Inject
     */
    protected $offerRepository;

    /**
     * Synchronize command.
     *
     * Updates local Offer database from API data source.
     *
     * @param bool $quiet
     * @return void
     */
    public function synchronizeCommand($quiet = false)
    {
        if (!$quiet) {
            $this->showProgressOnCommandLine();
        }

        $configuration = $this->getConfigurationForService('Tportal/Api');
        if ($this->hasClients($configuration)) {
            foreach ($this->getClients($configuration) as $clientConfiguration) {
                $this->synchronizeOffersFromTportal($clientConfiguration);
            }
        }

        $this->objectPathMappingService->flushMappings(Offer::class);
    }

    /**
     * Removes offers from the system according to the given clause.
     * If no WHERE clause is given then all offers will be deleted.
     *
     * # Remove all offers
     * $ php flow offer:remove
     *
     * # Remove offers according to WHERE clause
     * $ php flow offer:remove --where='offer.name LIKE "Segway%"'
     *
     * # Remove single offer
     * $ php flow offer:remove \
     *      --where="offer.Persistence_Object_Identifier='a4625eb4-6e84-4834-969d-c9f1d447408b'"
     *
     *
     * @param string|null $where DQL WHERE clause selecting the offers to delete.
     * @throws \Neos\Flow\Persistence\Exception\IllegalObjectTypeException
     */
    public function removeCommand(string $where = null)
    {
        $query = $this->offerRepository->createQueryBuilder('offer');
        if ($where) {
            $query->where($where);
            $this->outputLine('Deleting offers WHERE ' . $where);
        } else {
            $this->outputLine('Deleting all offers');
        }

        $count = (clone $query)->select('COUNT(offer) AS count')
                ->getQuery()
                ->execute([], AbstractQuery::HYDRATE_ARRAY)[0]['count'] ?? 0;
        $this->askForConfirmationAndAbortIfNoneGiven(sprintf('Do you really want to remove %d offers?', $count));

        $this->output->progressStart($count);
        foreach ($query->getQuery()->execute() as $offer) {
            $this->offerRepository->remove($offer);
            $this->output->progressAdvance();
        }

        $this->output->progressFinish();
    }



    private function synchronizeOffersFromTportal(array $configuration)
    {
        $this->emitStart(static::TYPE_TPORTAL);

        /** @var ApiService $service */
        $service = ServiceFactory::get('Tportal/Api', $configuration['baseUri'] ?? null);
        $service->setClientName($configuration['client']);

        $importer = new OfferImporter();
        $processed = 0;
        $service->fetch('offers', $this->wrapImportClosure(
            function (OfferAdapterInterface $record) use ($importer, &$processed) {
                $this->emitProgress(static::TYPE_TPORTAL, ++$processed);
                $importer->import($record);
            }
        ));

        $this->emitEnd(static::TYPE_TPORTAL);
    }
}
