<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Filter;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

/**
 * Event date filter class.
 *
 * Contrary to the event filter, filtering based on specific date values
 * is more accurate, and provides better usability in cases where you
 * want to list upcoming event dates rather than events.
 */
class EventDateFilter extends AbstractFilter
{
    /**
     * @var string
     */
    const DATE_FORMAT = 'Y-m-d';

    /**
     * @var array
     */
    protected $categories = [];

    /**
     * @var array
     */
    protected $eventTags = [];

    /**
     * Queries the from-date as minimum value.
     *
     * @var \DateTime|null
     */
    protected $fromDate;

    /**
     * Queries the from-date as maximum value.
     * @var \DateTime|null
     */
    protected $fromMaxDate;

    /**
     * @var \DateTime|null
     */
    protected $toDate;

    /**
     * @var bool|null
     */
    protected $highlight;

    /**
     * @var array
     */
    protected $locationZips = [];

    /**
     * @var array
     */
    protected $scope = [];

    /**
     * Initializes filter options.
     *
     * @param array $config
     * @return EventDateFilter
     */
    public function initialize(array $config)
    {
        parent::initialize($config);

        if (array_key_exists('categories', $config)) {
            $this->setCategories($config['categories']);
        }
        if (array_key_exists('eventTags', $config)) {
            $this->setEventTags($config['eventTags']);
        }
        if (array_key_exists('fromDate', $config)) {
            $this->setFromDate($this->asDateTime($config['fromDate']));
        } else {
            // ignore past events
            $this->setFromDate($this->asDateTime('now'));
        }
        if (array_key_exists('toDate', $config)) {
            $this->setToDate($this->asDateTime($config['toDate']));
        }
        if (array_key_exists('locationZips', $config)) {
            $this->setLocationZips($config['locationZips']);
        }
        if (array_key_exists('eventScope', $config)) {
            $this->setScope($config['eventScope']);
        }
        if (array_key_exists('highlight', $config)) {
            $this->setHighlight((bool) $config['highlight']);
        }
        if (array_key_exists('fromMaxDate', $config)) {
            $this->setFromMaxDate($config['fromMaxDate']);
        }

        return $this;
    }

    /**
     * @return array
     */
    public function getLocationZips(): array
    {
        return $this->locationZips;
    }

    /**
     * @param array $zips
     * @return EventDateFilter
     */
    public function setLocationZips(array $zips): self
    {
        $this->locationZips = $zips;
        return $this;
    }

    /**
     * @return \DateTime|null
     */
    public function getFromDate()
    {
        return $this->fromDate;
    }

    /**
     * @param \DateTime $date
     * @return self
     */
    public function setFromDate(\DateTime $date)
    {
        $this->fromDate = $date;
        return $this;
    }

    /**
     * @return \DateTime|null
     */
    public function getToDate()
    {
        return $this->toDate;
    }

    /**
     * @param \DateTime $date
     * @return self
     */
    public function setToDate(\DateTime $date)
    {
        $this->toDate = $date;
        return $this;
    }

    /**
     * @return bool|null
     */
    public function getHighlight()
    {
        return $this->highlight;
    }

    /**
     * @param bool|null $highlight
     * @return self
     */
    public function setHighlight($highlight)
    {
        $this->highlight = $highlight;
        return $this;
    }

    /**
     * @return \DateTime|null
     */
    public function getFromMaxDate()
    {
        return $this->fromMaxDate;
    }

    /**
     * @param \DateTime $date
     * @return self
     */
    public function setFromMaxDate(\DateTime $date): self
    {
        $this->fromMaxDate = $date;
        return $this;
    }

    /**
     * @return string[]
     */
    public function getCategories()
    {
        return $this->categories;
    }

    /**
     * @param string[] $categories
     * @return self
     */
    public function setCategories(array $categories)
    {
        $this->categories = $categories;
        return $this;
    }

    /**
     * @return int[]
     */
    public function getScope(): array
    {
        return $this->scope;
    }

    public function setScope(array $scope)
    {
        $this->scope = $scope;
        return $this;
    }

    /**
     * @return string[]
     */
    public function getEventTags()
    {
        return $this->eventTags;
    }

    /**
     * @param string[] $eventTags
     * @return self
     */
    public function setEventTags(array $eventTags)
    {
        $this->eventTags = $eventTags;
        return $this;
    }
}
