<?php
namespace Newland\Toubiz\Sync\Neos\Command;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\AbstractQuery;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cache\CacheManager;
use Neos\Flow\Core\Booting\Scripts;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\Service\Outdooractive;
use Newland\Toubiz\Api\Service\Outdooractive\ObjectAdapter\TourAdapter;
use Newland\Toubiz\Api\Service\ServiceFactory;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DbService;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DirectMarketerApiService;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\GastronomyApiService;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DbService\PointOfInterestAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DirectMarketerApiService\DirectMarketerAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService\GastronomyAdapter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;
use Newland\Toubiz\Sync\Neos\Enum\ArticleType;
use Newland\Toubiz\Sync\Neos\Domain\Repository\GeometryRepository;
use Newland\Toubiz\Sync\Neos\Importer\ArticleImporter;

/**
 * Articles command controller.
 *
 * Provides commands to manipulate article data.
 *
 * @Flow\Scope("singleton")
 */
class ArticlesCommandController extends AbstractCommandController
{
    protected static $mainTypeMap = [
        ArticleType::POI => ArticleConstants::TYPE_ATTRACTION,
        ArticleType::GASTRONOMY => ArticleConstants::TYPE_GASTRONOMY,
        ArticleType::TOURS => ArticleConstants::TYPE_TOUR,
        ArticleType::LODGINGS => ArticleConstants::TYPE_LODGING,
        ArticleType::DIRECT_MARKETERS => ArticleConstants::TYPE_DIRECT_MARKETER,
    ];

    const TYPE_GENERATE = 'generate';

    /**
     * @var CacheManager
     * @Flow\Inject
     */
    protected $cacheManager;

    /**
     * @var ArticleRepository
     * @Flow\Inject
     */
    protected $articleRepository;

    /**
     * @var GeometryRepository
     * @Flow\Inject
     */
    protected $geometryRepository;

    /**
     * @Flow\InjectConfiguration(package="Neos.Flow")
     * @var array
     */
    protected $flowSettings;

    /**
     * Synchronize command.
     *
     * Updates local articles database from API data source.
     *
     * @param bool $quiet
     * @param string $only Can be one article type of [poi, gastronomy, lodgings, tours, directMarketers]
     * @param int $minutesToLive The age of records in minutes after which they are considered "old" and will be
     *     deleted. Default is 7 days (=10080 minutes) ago.
     * @return void
     */
    public function synchronizeCommand($quiet = false, string $only = '', int $minutesToLive = 10080)
    {
        if (!$quiet) {
            $this->showProgressOnCommandLine();
        }
        $this->importGeometry();
        $articleTypes = ArticleType::values();
        if ($only !== '') {
            $articleTypes = [ $only ];
        }

        foreach ($articleTypes as $currentArticleType) {
            $this->synchronizeArticleType($currentArticleType);
            $mainType = static::$mainTypeMap[$currentArticleType];
            $deletedCount = $this->deleteOldRecords($minutesToLive, $mainType);
            $this->outputLine(sprintf("\nDeleted %d %s.", $deletedCount, $currentArticleType));
        }

        $this->objectPathMappingService->flushMappings(Article::class);
        $this->rebuildUrls();
        $this->cleanup();
    }

    /**
     * Removes articles from the system according to the given clause.
     * If no WHERE clause is given then all articles will be deleted.
     *
     * # Remove all articles
     * $ php flow articles:remove
     *
     * # Remove articles according to WHERE clause
     * $ php flow articles:remove --where='article.client="westlicherbodensee"'
     *
     * # Remove single article
     * $ php flow articles:remove \
     *      --where="article.Persistence_Object_Identifier='a4625eb4-6e84-4834-969d-c9f1d447408b'"
     *
     *
     * @param string|null $where DQL WHERE clause selecting the articles to delete.
     */
    public function removeCommand(string $where = null): void
    {
        $query = $this->articleRepository->createQueryBuilder('article');
        if ($where) {
            $query->where($where);
            $this->outputLine('Deleting articles WHERE ' . $where);
        } else {
            $this->outputLine('Deleting all articles');
        }

        $count = (clone $query)->select('COUNT(article) AS count')
                ->getQuery()
                ->execute([], AbstractQuery::HYDRATE_ARRAY)[0]['count'] ?? 0;
        $this->askForConfirmationAndAbortIfNoneGiven(sprintf('Do you really want to remove %d articles?', $count));
        $this->output->progressStart($count);

        foreach ($query->getQuery()->execute() as $article) {
            $this->articleRepository->remove($article);
            $this->output->progressAdvance();
        }

        $this->output->progressFinish();
    }

    /**
     * Synchronizes lodging data as articles from TPortal.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizeLodgingsFromTportal(array $configuration)
    {
        $this->emitStart(ArticleType::LODGINGS);
        $client = $configuration['client'];

        /** @var \Newland\Toubiz\Api\Service\Tportal\ApiService $service */
        $service = ServiceFactory::get('Tportal/Api', $configuration['baseUri'] ?? null);
        $service->setClientName($client);
        $service->setLogger($this->logger);

        $processed = 0;
        $service->fetch(
            'lodgings',
            $this->wrapImportClosure(
                function ($record) use ($client, &$processed) {
                    $this->emitProgress(ArticleType::LODGINGS, ++$processed);
                    $importer = new ArticleImporter;
                    $importer->setClient($client);
                    $importer->import($record);
                }
            )
        );

        $this->emitEnd(ArticleType::LODGINGS);
    }

    /**
     * Synchronizes POI data from legacy toubiz DB service.
     *
     * @param array $configuration
     * @param string $language
     * @return void
     */
    protected function synchronizePointOfInterestsFromDbService($configuration, string $language)
    {
        $this->emitStart(ArticleType::POI, compact('language'));

        /** @var DbService $service */
        $service = ServiceFactory::get('Toubiz/Legacy/Db', $configuration['baseUri'] ?? null);
        $service->setLogger($this->logger);
        $service->setClientName($configuration['client']);
        $service->setApiKey($configuration['apiKey']);
        $service->setLanguage($language);

        $minLevelOfMaintenance = $configuration['minLevelOfMaintenance'] ?? 0;

        $processed = 0;
        $service->fetch(
            'pointOfInterests',
            $this->wrapImportClosure(
                function (PointOfInterestAdapter $record) use ($minLevelOfMaintenance, &$processed, $language) {
                    $this->emitProgress(ArticleType::POI, ++$processed, compact('language'));
                    if ($record->getLevelOfMaintenance() > $minLevelOfMaintenance) {
                        (new ArticleImporter())->import($record);
                    } else {
                        (new ArticleImporter())->remove($record);
                    }
                }
            )
        );

        $this->emitEnd(ArticleType::POI, compact('language'));
    }

    /**
     * Synchronizes gastronomy data from legacy API.
     *
     * @param array $configuration
     * @param string $language
     * @return void
     */
    protected function synchronizeGastronomyFromApi($configuration, string $language)
    {
        $this->emitStart(ArticleType::GASTRONOMY, compact('language'));

        /** @var GastronomyApiService $service */
        $service = ServiceFactory::get('Toubiz/Legacy/GastronomyApi', $configuration['baseUri'] ?? null);
        $service->setApiKey($configuration['apiKey']);
        $service->setLanguage($language);

        $processed = 0;
        $service->fetch(
            'gastronomy',
            $this->wrapImportClosure(
                function (GastronomyAdapter $record) use (&$processed, $language) {
                    $this->emitProgress(ArticleType::GASTRONOMY, ++$processed, compact('language'));
                    if ($record->getOnlineStatus()) {
                        (new ArticleImporter())->import($record);
                    } else {
                        (new ArticleImporter())->remove($record);
                    }
                }
            )
        );

        $this->emitEnd(ArticleType::GASTRONOMY, compact('language'));
    }

    /**
     * Synchronizes direct marketers from toubiz legacy API.
     *
     * @param array $configuration
     * @param string $language
     * @return void
     */
    protected function synchronizeDirectMarketersFromApi(array $configuration, string $language)
    {
        $this->emitStart(ArticleType::DIRECT_MARKETERS, compact('language'));

        /** @var DirectMarketerApiService $service */
        $service = ServiceFactory::get('Toubiz/Legacy/DirectMarketerApi', $configuration['baseUri'] ?? null);
        $service->setApiKey($configuration['apiKey']);
        $service->setLanguage($language);

        $processed = 0;
        $service->fetch(
            'directMarketers',
            $this->wrapImportClosure(
                function (DirectMarketerAdapter $record) use (&$processed, $language) {
                    $this->emitProgress(ArticleType::DIRECT_MARKETERS, ++$processed, compact('language'));
                    $importer = new ArticleImporter;
                    if ($record->getOnlineStatus()) {
                        $importer->setOpeningTimesFormat(ArticleConstants::OPENING_TIMES_FORMAT_LEGACY);
                        $importer->import($record);
                    } else {
                        $importer->remove($record);
                    }
                }
            )
        );

        $this->emitEnd(ArticleType::DIRECT_MARKETERS, compact('language'));
    }

    /**
     * Synchronizes tours.
     *
     * @param array $configuration
     * @param string $language
     * @return void
     */
    protected function synchronizeToursFromApi(array $configuration, string $language)
    {

        /** @var Outdooractive\ApiService $service */
        $service = ServiceFactory::get('Outdooractive/Api', $configuration['baseUri'] ?? null);
        $service->setClientName($configuration['client']);
        $service->setApiKey($configuration['apiKey']);
        $service->setLanguage($language);

        $this->emitStart(ArticleType::TOURS, compact('language'));

        $processed = 0;
        $service->fetch(
            'tours',
            $this->wrapImportClosure(
                function (TourAdapter $tour) use (&$processed, $language) {
                    $this->emitProgress(ArticleType::TOURS, ++$processed, compact('language'));
                    (new ArticleImporter())->import($tour);
                }
            )
        );

        $this->emitEnd(ArticleType::TOURS, compact('language'));
    }

    private function synchronizeArticleType(string $type): void
    {
        if (ArticleType::validValue($type) === false) {
            $message = sprintf('Unknown article type "%s". Valid types are: ', $type)
                . implode(', ', ArticleType::values());
            throw new \Exception($message);
        }

        $method = 'synchronizeArticleType' . ucfirst($type);
        $this->$method();
    }

    private function deleteOldRecords(int $minutesToLive, int $mainType = null): int
    {
        $importer = new ArticleImporter;
        $timeToLive = new \DateInterval(sprintf('PT%dM', $minutesToLive));
        return $importer->deleteOldRecords($timeToLive, $mainType);
    }

    /** @noinspection PhpUnusedPrivateMethodInspection */
    private function synchronizeArticleTypeLodgings(): void
    {
        $configuration = $this->getConfigurationForService('Tportal/Api');
        if ($this->hasClients($configuration)) {
            foreach ($this->getClients($configuration) as $clientConfiguration) {
                $this->synchronizeLodgingsFromTportal($clientConfiguration);
            }
        }
    }

    /** @noinspection PhpUnusedPrivateMethodInspection */
    private function synchronizeArticleTypePoi(): void
    {
        $configuration = $this->getConfigurationForService('Toubiz/Legacy/Db');
        if ($configuration) {
            foreach ($configuration['languages'] as $language) {
                $this->synchronizePointOfInterestsFromDbService($configuration, $language);
            }
        }
    }

    /** @noinspection PhpUnusedPrivateMethodInspection */
    private function synchronizeArticleTypeGastronomy(): void
    {
        $configuration = $this->getConfigurationForService('Toubiz/Legacy/GastronomyApi');
        if ($configuration) {
            foreach ($configuration['languages'] as $language) {
                $this->synchronizeGastronomyFromApi($configuration, $language);
            }
        }
    }

    /** @noinspection PhpUnusedPrivateMethodInspection */
    private function synchronizeArticleTypeDirectMarketers(): void
    {
        $configuration = $this->getConfigurationForService('Toubiz/Legacy/DirectMarketerApi');
        if ($configuration) {
            foreach ($configuration['languages'] as $language) {
                $this->synchronizeDirectMarketersFromApi($configuration, $language);
            }
        }
    }

    /** @noinspection PhpUnusedPrivateMethodInspection */
    private function synchronizeArticleTypeTours(): void
    {
        $configuration = $this->getConfigurationForService('Outdooractive/Api');
        if ($configuration) {
            foreach ($configuration['languages'] as $language) {
                $this->synchronizeToursFromApi($configuration, $language);
            }
        }
    }

    private function cleanup()
    {
        $this->outputLine('Flushing cache');
        $this->cacheManager->flushCaches();
        Scripts::executeCommand('neos.flow:cache:flush', $this->flowSettings);
        $this->outputLine("\rImport finished");
    }

    private function importGeometry()
    {
        if ($this->geometryRepository->isEmpty()) {
            $this->outputLine('Geometries not found, running geometry:seedzippolygons command');
            Scripts::executeCommand('newland.toubiz.sync.neos:geometry:seedzippolygons', $this->flowSettings);

            $this->outputLine("\nGeometries Imported, Continuing...");
        } else {
            $this->outputLine("\nGeometries Found, Continuing...");
        }
    }

    private function rebuildUrls()
    {
        $this->outputLine("\nGenerating URL Map");
        $this->emitEnd(static::TYPE_GENERATE);
        $this->outputLine("\nURL Map generated");
    }
}
