<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Filter;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\PersistentCollection;

/**
 * Article filter class.
 */
class ArticleFilter extends AbstractFilter
{
    /**
     * @var array
     */
    protected $categories = [];

    /**
     * Field to match for categories.
     *
     * Default ist persistence_object_identifier, but e.g. the
     * category mapping returns originalId values.
     *
     * @var string
     */
    protected $categoriesIdentifierField = 'Persistence_Object_Identifier';

    /**
     * @var float|null
     */
    protected $latitude;

    /**
     * @var float|null
     */
    protected $longitude;

    /**
     * @var int|null
     */
    protected $mainType;

    /**
     * @var string|null
     */
    protected $currentSite = '';

    /**
     * Precision for lat/long comparison.
     * - 1 => ~11.1km
     * - 2 => ~1.1km
     * - 3 => ~110m
     * - 4 => ~11m
     *
     * @var integer|null
     */
    protected $coordinatePrecision;

    /**
     * @var array
     */
    protected $zips = [];

    /**
     * @var bool
     */
    protected $excludeUnsafeCoordinates = false;

    /**
     * @var string[]
     */
    protected $identifiers = [];

    /**
     * @var array
     */
    protected $attributesIn = [];

    /**
     * @var array
     */
    protected $attributesRange = [];

    /**
     * Initializes filter options.
     *
     * @param array $config
     * @return self
     */

    public function initialize(array $config)
    {
        parent::initialize($config);
        return $this;
    }

    /**
     * @param array $categories
     * @return self
     */
    public function setCategories(array $categories)
    {
        $this->categories = $categories;
        return $this;
    }

    /**
     * @param PersistentCollection $categories
     * @return self
     */
    public function setCategoryCollection(PersistentCollection $categories)
    {
        $this->categories = [];
        foreach ($categories as $category) {
            $this->categories[] = $category->getPersistenceObjectIdentifier();
        }
        return $this;
    }

    /**
     * @return array
     */
    public function getCategories()
    {
        return $this->categories;
    }

    /**
     * @param string $fieldName
     * @return self
     */
    public function setCategoriesIdentifierField($fieldName)
    {
        $this->categoriesIdentifierField = $fieldName;
        return $this;
    }

    /**
     * @return string
     */
    public function getCategoriesIdentifierField()
    {
        return $this->categoriesIdentifierField;
    }

    /**
     * @param float|null $latitude
     * @return self
     */
    public function setLatitude($latitude)
    {
        $this->latitude = $latitude;
        return $this;
    }

    /**
     * @return float|null
     */
    public function getLatitude()
    {
        if ($this->coordinatePrecision !== null) {
            return round((float) $this->latitude, $this->coordinatePrecision);
        }
        return $this->latitude;
    }

    /**
     * @param float|null $longitude
     * @return self
     */
    public function setLongitude($longitude)
    {
        $this->longitude = $longitude;
        return $this;
    }

    /**
     * @return float|null
     */
    public function getLongitude()
    {
        if ($this->coordinatePrecision !== null) {
            return round((float) $this->longitude, $this->coordinatePrecision);
        }
        return $this->longitude;
    }

    /**
     * @var integer $precision
     * @return self
     */
    public function setCoordinatePrecision($precision)
    {
        $this->coordinatePrecision = $precision;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getCoordinatePrecision()
    {
        return $this->coordinatePrecision;
    }

    /**
     * @param int $mainType
     * @return self
     */
    public function setMainType($mainType)
    {
        $this->mainType = $mainType;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getMainType()
    {
        return $this->mainType;
    }

    /**
     * @return array
     */
    public function getZips()
    {
        return $this->zips;
    }

    /**
     * @param array $zips
     */
    public function setZips(array $zips)
    {
        $this->zips = $zips;
    }

    /**
     * @return bool
     */
    public function getExcludeUnsafeCoordinates(): bool
    {
        return $this->excludeUnsafeCoordinates;
    }

    /**
     * @param bool $excludeUnsafeCoordinates
     */
    public function setExcludeUnsafeCoordinates(bool $excludeUnsafeCoordinates)
    {
        $this->excludeUnsafeCoordinates = $excludeUnsafeCoordinates;
    }

    public function setIdentifiers(array $identifiers)
    {
        $this->identifiers = $identifiers;
    }

    public function getIdentifiers(): array
    {
        return $this->identifiers;
    }

    public function setCurrentSite(string $currentSite)
    {
        $this->currentSite = $currentSite;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getCurrentSite()
    {
        return $this->currentSite;
    }

    public function setAttributesIn(array $attributes)
    {
        $this->attributesIn = $attributes;
    }

    public function getAttributesIn()
    {
        return $this->attributesIn;
    }

    public function setAttributesRange(array $attributes)
    {
        $this->attributesRange = $attributes;
    }

    public function getAttributesRange()
    {
        return $this->attributesRange;
    }
}
