<?php declare(strict_types=1);

namespace Newland\Toubiz\Sync\Neos\Tests\Unit\Importer;

use Neos\Flow\Tests\FunctionalTestCase;
use Newland\Toubiz\Sync\Neos\Importer\EventDateImporter;
use Newland\Toubiz\Sync\Neos\Tests\Factory\EventDateFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\EventFactory;
use Newland\Toubiz\Sync\Neos\Tests\Unit\Importer\Mock\EventDateAdapterMock;

class EventDateImporterTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;

    /** @var EventDateImporter */
    protected $eventDateImporter;

    /** @var EventDateFactory */
    protected $factory;

    public function setUp(): void
    {
        parent::setUp();
        $this->eventDateImporter = $this->objectManager->get(EventDateImporter::class);
        $this->eventDateImporter->setEvent((new EventFactory($this->objectManager))->create());
        $this->factory = new EventDateFactory($this->objectManager);
    }

    public function testReusesGivenModelIfSpecified(): void
    {
        $date = $this->factory->create();
        $imported = $this->eventDateImporter->import(new EventDateAdapterMock([]), $date);

        self::assertEquals($imported->getPersistenceObjectIdentifier(), $date->getPersistenceObjectIdentifier());
    }

    public function testUsingSameExternalIdAsExistingModelDoesNotAutomaticallyReuseIt(): void
    {
        $date = $this->factory->create([ 'language' => 'de' ]);
        $this->eventDateImporter->setLanguage('en');
        $this->eventDateImporter->setEvent($date->getEvent());
        $imported = $this->eventDateImporter->import(
            new EventDateAdapterMock([ 'externalId' => $date->getOriginalId() ]),
            null
        );

        self::assertNotEquals($imported->getPersistenceObjectIdentifier(), $date->getPersistenceObjectIdentifier());
    }

    public function testUsingSameExternalIdDoesNotYieldSameModel(): void
    {
        $this->eventDateImporter->setLanguage('de');
        $first = $this->eventDateImporter->import(new EventDateAdapterMock([ 'externalId' => 'foo' ]), null);
        $this->eventDateImporter->setLanguage('en');
        $second = $this->eventDateImporter->import(new EventDateAdapterMock([ 'externalId' => 'foo' ]), null);

        self::assertNotEquals($first->getPersistenceObjectIdentifier(), $second->getPersistenceObjectIdentifier());
    }

    public function testMapsDates(): void
    {
        $adapter = new EventDateAdapterMock(
            [
                'externalId' => 'foo',
                'beginsAt' => new \DateTime('2020-01-01T15:00:00'),
                'beginsAtSpecificTime' => true,
                'endsAt' => new \DateTime('2020-01-01T23:59:59'),
                'endsAtSpecificTime' => false,
            ]
        );

        $date = $this->eventDateImporter->import($adapter, null);

        self::assertTrue($date->getBeginsAtSpecificTime());
        self::assertFalse($date->getEndsAtSpecificTime());
        self::assertEquals('2020-01-01 15:00:00', $date->getBeginsAt()->format('Y-m-d H:i:s'));
        self::assertEquals('2020-01-01 23:59:59', $date->getEndsAt()->format('Y-m-d H:i:s'));
    }
}
