<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Repository;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\QueryBuilder;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Sync\Neos\Domain\Filter\EventDateFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\FilterInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\EventDate;

/**
 * Event date repository.
 *
 * @Flow\Scope("singleton")
 *
 * @method EventDate findOneByOriginalId(string $originalId)
 */
class EventDateRepository extends AbstractRepository
{


    /**
     * Deletes records from the past.
     *
     * @return void
     */
    public function deletePastRecords()
    {
        $endsAt = new \DateTime();
        $endsAt->setTime(0, 0, 0);
        $endsAt->modify('-7 days');

        $query = $this->createQueryBuilder($this->getTableAlias());
        $query->delete()
              ->where('eventDate.endsAt < :ends_at')
              ->setParameter('ends_at', $endsAt);

        $query->getQuery()->execute();
    }

    /**
     * Applies filter functions (from the event date filter) onto
     * the given query builder.
     *
     * @param EventDateFilter|FilterInterface $filter
     * @param QueryBuilder $query
     * @return QueryBuilder
     */
    protected function applyFilter(FilterInterface $filter, QueryBuilder $query): QueryBuilder
    {
        parent::applyBasicFilter($filter, $query);
        $query->innerJoin('eventDate.event', 'event');

        if ($filter->hasCategories() || $filter->getHighlight()) {
            $query->innerJoin('event.categories', 'categories');
        }

        if ($filter->hasCategories()) {
            $query->andWhere('categories.Persistence_Object_Identifier IN (:categories)')
                  ->setParameter('categories', $filter->getCategories());
        }

        if ($filter->hasFromDate()) {
            $query->andWhere('DATE(eventDate.beginsAt) >= :fromDate')
                  ->setParameter('fromDate', $filter->getFromDate()->format('Y-m-d'));
        }

        if ($filter->hasFromMaxDate()) {
            $query->andWhere('DATE(eventDate.beginsAt) <= :fromMaxDate')
                  ->setParameter('fromMaxDate', $filter->getFromMaxDate()->format('Y-m-d'));
        }

        if ($filter->hasToDate()) {
            $query->andWhere('DATE(eventDate.endsAt) <= :date')
                  ->setParameter('date', $filter->getToDate()->format('Y-m-d'));
        }

        if ($filter->getHighlight()) {
            $query->andWhere('event.isHighlight = :flag')
                  ->setParameter('flag', $filter->getHighlight());
        }

        if ($filter->hasLocationZips()) {
            $query->innerJoin('event.location', 'location');
            $query->andWhere($query->expr()->in('location.zip', $filter->getLocationZips()));
        }

        if ($filter->hasScope()) {
            $query->andWhere($query->expr()->in('event.scope', $filter->getScope()));
        }

        return $query;
    }
}
