<?php
namespace Newland\Toubiz\Sync\Neos\Command;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\Service\ServiceFactory;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\EventApiService;
use Newland\Toubiz\Sync\Neos\Importer\EventImporter;

/**
 * Events command controller.
 *
 * Provides commands to manipulate events data.
 *
 * @Flow\Scope("singleton")
 */
class EventsCommandController extends AbstractCommandController
{
    const TYPE_LEGACY_TOUBIZ = 'LegacyToubiz';

    /**
     * Synchronize command.
     *
     * Updates local events database from API data source.
     *
     * @todo delete persisted events not being in the API response
     * @param bool $quiet
     * @return void
     * @throws \Exception
     */
    public function synchronizeCommand($quiet = false)
    {
        if (!$quiet) {
            $this->showProgressOnCommandLine();
        }

        // There is currently only one API providing events.
        $configuration = $this->getConfigurationForService('Toubiz/Legacy/EventApi');
        if (!$configuration) {
            throw new \Exception('Toubiz/Legacy/EventApi service not configured!');
        }
        $this->synchronizeEventsFromLegacyToubiz($configuration);
        (new EventImporter())->postImport();
    }


    private function synchronizeEventsFromLegacyToubiz(array $configuration)
    {
        $this->emitStart(static::TYPE_LEGACY_TOUBIZ);

        /** @var EventApiService $toubizService */
        $toubizService = ServiceFactory::get('Toubiz/Legacy/EventApi');
        $toubizService->setClientName($configuration['client']);
        $toubizService->setApiKey($configuration['apiKey']);

        if (array_key_exists('organizer', $configuration)) {
            $toubizService->setOrganizerId($configuration['organizer']);
        }

        $eventList = $toubizService->fetchActiveEvents();
        if (!$eventList) {
            throw new \Exception('No event data received for synchronizing!');
        }

        foreach ($eventList as $i => $listItem) {
            $this->emitProgress(static::TYPE_LEGACY_TOUBIZ, $i);
            $importer = new EventImporter;
            $eventDetail = $toubizService->fetchEventDetails($listItem->getExternalId());
            $importer->import($eventDetail);
        }
        $this->emitEnd(static::TYPE_LEGACY_TOUBIZ);
    }
}
