<?php
namespace Newland\Toubiz\Sync\Neos\Command;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\Service\ServiceFactory;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DbService\PointOfInterestAdapter;
use Newland\Toubiz\Sync\Neos\Importer\ArticleImporter;

/**
 * Articles command controller.
 *
 * Provides commands to manipulate article data.
 *
 * @Flow\Scope("singleton")
 */
class ArticlesCommandController extends AbstractCommandController
{
    const TYPE_LODGINGS = 'lodgings';
    const TYPE_POI = 'poi';
    const TYPE_GASTRONOMY = 'gastronomy';
    const TYPE_DIRECT_MARKETERS = 'directMarketers';
    const TYPE_TOURS = 'tours';

    /**
     * Synchronize command.
     *
     * Updates local articles database from API data source.
     *
     * @param bool $quiet
     * @param int $minutesToLive The age of records in minutes after which they are considered "old" and will be
     *     deleted. Default is 7 days (=10080 minutes) ago.
     * @return void
     */
    public function synchronizeCommand($quiet = false, $minutesToLive = 10080)
    {
        if (!$quiet) {
            $this->showProgressOnCommandLine();
        }

        $configuration = $this->getConfigurationForService('Tportal/Api');
        if ($this->hasClients($configuration)) {
            foreach ($this->getClients($configuration) as $clientConfiguration) {
                $this->synchronizeLodgingsFromTportal($clientConfiguration);
            }
        }

        $configuration = $this->getConfigurationForService('Toubiz/Legacy/Db');
        if ($configuration) {
            $this->synchronizePointOfInterestsFromDbService($configuration);
        }

        $configuration = $this->getConfigurationForService('Toubiz/Legacy/GastronomyApi');
        if ($configuration) {
            $this->synchronizeGastronomyFromApi($configuration);
        }

        $configuration = $this->getConfigurationForService('Toubiz/Legacy/DirectMarketerApi');
        if ($configuration) {
            $this->synchronizeDirectMarketersFromApi($configuration);
        }

        $configuration = $this->getConfigurationForService('Outdooractive/Api');
        if ($configuration) {
            $this->synchronizeToursFromApi($configuration);
        }

        $importer = new ArticleImporter;
        $timeToLive = new \DateInterval(sprintf('PT%dM', (int) $minutesToLive));
        $importer->deleteOldRecords($timeToLive);
    }

    /**
     * Synchronize tours command.
     *
     * Updates local database with tour articles from API data source.
     *
     * @param bool $quiet
     * @param int $minutesToLive The age of records in minutes after which they are considered "old" and will be
     *     deleted. Default is 7 days (=10080 minutes) ago.
     * @param null|int $limit
     * @return void
     * @throws \Exception
     */
    public function synchronizeToursCommand($quiet = false, $minutesToLive = 10080, $limit = null)
    {
        if (!$quiet) {
            $this->showProgressOnCommandLine();
        }

        $configuration = $this->getConfigurationForService('Outdooractive/Api');
        if ($configuration) {
            $this->synchronizeToursFromApi($configuration, $limit);
        }

        $importer = new ArticleImporter;
        $timeToLive = new \DateInterval(sprintf('PT%dM', (int) $minutesToLive));
        $importer->deleteOldRecords($timeToLive);
    }

    /**
     * Synchronizes lodging data as articles from TPortal.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizeLodgingsFromTportal($configuration)
    {
        $this->emitStart(static::TYPE_LODGINGS);
        $client = $configuration['client'];

        /** @var \Newland\Toubiz\Api\Service\Tportal\ApiService $service */
        $service = ServiceFactory::get('Tportal/Api');
        $service->setClientName($client);

        $processed = 0;
        $service->fetch(
            'lodgings',
            function ($record) use ($client, &$processed) {
                $this->emitProgress(static::TYPE_LODGINGS, ++$processed);
                $importer = new ArticleImporter;
                $importer->setClient($client);
                $importer->import($record);
            }
        );

        $this->emitEnd(static::TYPE_LODGINGS);
    }

    /**
     * Synchronizes POI data from legacy toubiz DB service.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizePointOfInterestsFromDbService($configuration)
    {
        $this->emitStart(static::TYPE_POI);

        $service = ServiceFactory::get('Toubiz/Legacy/Db');
        $service->setClientName($configuration['client']);
        $service->setApiKey($configuration['apiKey']);

        $minLevelOfMaintenance = $configuration['minLevelOfMaintenance'] ?? 0;

        $processed = 0;
        $service->fetch(
            'pointOfInterests',
            function (PointOfInterestAdapter $record) use ($minLevelOfMaintenance, &$processed) {
                $this->emitProgress(static::TYPE_POI, ++$processed);
                if ($record->getLevelOfMaintenance() > $minLevelOfMaintenance) {
                    (new ArticleImporter())->import($record);
                } else {
                    (new ArticleImporter())->remove($record);
                }
            }
        );

        $this->emitEnd(static::TYPE_POI);
    }

    /**
     * Synchronizes gastronomy data from legacy API.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizeGastronomyFromApi($configuration)
    {
        $this->emitStart(static::TYPE_GASTRONOMY);

        $service = ServiceFactory::get('Toubiz/Legacy/GastronomyApi');
        $service->setApiKey($configuration['apiKey']);

        $processed = 0;
        $service->fetch(
            'gastronomy',
            function ($record) use (&$processed) {
                $this->emitProgress(static::TYPE_GASTRONOMY, ++$processed);
                $importer = new ArticleImporter;
                $importer->import($record);
            }
        );

        $this->emitEnd(static::TYPE_GASTRONOMY);
    }

    /**
     * Synchronizes direct marketers from toubiz legacy API.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizeDirectMarketersFromApi(array $configuration)
    {
        $this->emitStart(static::TYPE_DIRECT_MARKETERS);

        $service = ServiceFactory::get('Toubiz/Legacy/DirectMarketerApi');
        $service->setApiKey($configuration['apiKey']);

        $processed = 0;
        $service->fetch(
            'directMarketers',
            function ($record) use (&$processed) {
                $this->emitProgress(static::TYPE_DIRECT_MARKETERS, ++$processed);
                $importer = new ArticleImporter;
                $importer->setOpeningTimesFormat(ArticleConstants::OPENING_TIMES_FORMAT_LEGACY);
                $importer->import($record);
            }
        );

        $this->emitEnd(static::TYPE_DIRECT_MARKETERS);
    }

    /**
     * Synchronizes tours.
     *
     * @param array $configuration
     * @param null $limit
     * @return void
     * @throws \Exception
     */
    protected function synchronizeToursFromApi($configuration, $limit = null)
    {
        $this->emitStart(static::TYPE_TOURS);

        $service = ServiceFactory::get('Outdooractive/Api');
        $service->setClientName($configuration['client']);
        $service->setApiKey($configuration['apiKey']);

        $records = $service->fetchTours();
        if (!$records) {
            throw new \Exception('No tour data received for synchronizing!');
        }

        $limit = $limit ?? \count($records);

        for ($i = 0; $i < $limit; $i++) {
            $record = $records[$i];
            $this->emitProgress(static::TYPE_TOURS, $i);
            $importer = new ArticleImporter;
            $importer->import($record);
        }

        $this->emitEnd(static::TYPE_TOURS);
    }
}
