<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Repository;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\QueryBuilder;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Persistence\QueryResultInterface;
use Newland\Toubiz\Sync\Neos\Domain\Filter\AttributeFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\FilterInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Model\Attribute;

/**
 * Attribute repository.
 *
 * @Flow\Scope("singleton")
 *
 * @method Attribute findOneByOriginalId(string $originalId)
 */
class AttributeRepository extends AbstractRepository
{

    /**
     * @param Article $article
     * @param string $name
     * @return Attribute|null
     */
    public function findOneByArticleAndName(Article $article, string $name)
    {
        $attribute = null;

        $result = $this->createQueryBuilder('attribute')
            ->innerJoin('attribute.article', 'article')
            ->andWhere('article.originalId = :originalId')
            ->setParameter('originalId', $article->getOriginalId())
            ->andWhere('attribute.name = :name')
            ->setParameter('name', $name)
            ->setMaxResults(1)
            ->getQuery()
            ->execute();

        if (is_array($result) && \count($result)) {
            $attribute = $result[0];
        }
        return $attribute;
    }

    /**
     * Applies filter functions (from the attribute filter) onto
     * the given query builder.
     *
     * @param AttributeFilter $filter
     * @param QueryBuilder $query
     * @return QueryBuilder
     */
    protected function applyFilter(FilterInterface $filter, QueryBuilder $query): QueryBuilder
    {
        parent::applyBasicFilter($filter, $query);

        if ($filter->hasOriginalIds()) {
            $query->andWhere(
                $query->expr()->in('attribute.originalid', $filter->getOriginalIds())
            );
        }

        return $query;
    }
}
