<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Repository;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\QueryBuilder;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Persistence\QueryResultInterface;
use Newland\Toubiz\Sync\Neos\Command\OrphanFinder;
use Newland\Toubiz\Sync\Neos\Domain\Filter\ArticleFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\FilterInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;

/**
 * Article repository.
 *
 * @Flow\Scope("singleton")
 *
 * @method Article findOneByOriginalId(string $originalId)
 */
class ArticleRepository extends AbstractRepository implements OrphanFinder
{
    const MIN_LATITUDE = -90;
    const MAX_LATITUDE = 90;
    const MIN_LONGITUDE = -180;
    const MAX_LONGITUDE = 180;

    /**
     * Specific method for fetching all data for a data source
     * (e.g. multi-select in a neos node type property) in the
     * most performant way.
     *
     * @param ArticleFilter $filter
     * @return array
     */
    public function findAllForDataSource(ArticleFilter $filter)
    {
        $query = $this->createQueryBuilder('article');
        $query->select(
            [
                'article.name AS article_name',
                'article.Persistence_Object_Identifier AS article_identifier',
                'category.title AS category_title',

            ]
        )->join('article.categories', 'category')
            ->addOrderBy('category.title', 'asc')
            ->addOrderBy('article.name', 'asc');

        $this->articleClientFilterService->addClientWhereClause($query, $filter);

        return $query->getQuery()->getScalarResult();
    }

    /**
     * @param string $originalId
     * @param string $client
     * @return Article|null
     */
    public function findOneByOriginalIdAndClient(string $originalId, string $client)
    {
        $query = $this->createQueryBuilder('article');

        $result = $query->where($query->expr()->eq('article.originalId', ':originalId'))
            ->setParameter('originalId', $originalId)
            ->andWhere(
                $query->expr()->orX(
                    $query->expr()->eq('article.client', ':client'),
                    $query->expr()->eq('article.client', ':empty'),
                    $query->expr()->isNull('article.client')
                )
            )
            ->setParameter('client', $client)
            ->setParameter('empty', '')
            ->setMaxResults(1)
            ->getQuery()
            ->execute();

        return empty($result) ? null : $result[0];
    }

    /**
     * Applies filter functions (from the article filter) onto
     * the given query builder.
     *
     * @param ArticleFilter $filter
     * @param QueryBuilder $query
     * @return QueryBuilder
     */
    protected function applyFilter(FilterInterface $filter, QueryBuilder $query): QueryBuilder
    {
        parent::applyBasicFilter($filter, $query);

        // Category is always joined as e.g. sorting relies on it.
        $query->leftJoin('article.categories', 'category');
        $query->leftJoin('article.mainAddress', 'mainAddress');

        if ($filter->hasMainType()) {
            $query->andWhere($query->expr()->eq('article.mainType', $filter->getMainType()));
            if ($this->articleClientFilterService->needsFilter('mainType', $filter->getMainType(), $filter)) {
                $clientWhere = $this->articleClientFilterService->generateEqExprFor(
                    $query,
                    'mainType',
                    $filter->getMainType(),
                    $filter,
                    'article'
                );
                if ($clientWhere) {
                    $query->andWhere($clientWhere);
                }
            }
        } else {
            $this->articleClientFilterService->addClientWhereClause($query, $filter);
        }
        if ($filter->hasCategories()) {
            $query->andWhere(
                $query->expr()->in(
                    'category.' . $filter->getCategoriesIdentifierField(),
                    $filter->getCategories()
                )
            );
        }
        if ($filter->hasLatitude()) {
            $query->andWhere(
                $query->expr()->like(
                    'mainAddress.latitude',
                    $query->expr()->literal($filter->getLatitude() . '%')
                )
            );
        }
        if ($filter->hasLongitude()) {
            $query->andWhere(
                $query->expr()->like(
                    'mainAddress.longitude',
                    $query->expr()->literal($filter->getLongitude() . '%')
                )
            );
        }
        if ($filter->hasZips()) {
            $query->andWhere(
                $query->expr()->in(
                    'mainAddress.zip',
                    $filter->getZips()
                )
            );
        }

        if ($filter->getExcludeUnsafeCoordinates()) {
            $query->andWhere(
                $query->expr()->isNotNull('mainAddress.latitude'),
                $query->expr()->isNotNull('mainAddress.longitude'),
                $query->expr()->lte('mainAddress.latitude', static::MAX_LATITUDE),
                $query->expr()->gte('mainAddress.latitude', static::MIN_LATITUDE),
                $query->expr()->lte('mainAddress.longitude', static::MAX_LONGITUDE),
                $query->expr()->gte('mainAddress.longitude', static::MIN_LONGITUDE),
                $query->expr()->neq('mainAddress.latitude', 0),
                $query->expr()->neq('mainAddress.longitude', 0)
            );
        }

        if ($filter->hasIdentifiers()) {
            $query->andWhere(
                $query->expr()->in(
                    'article.Persistence_Object_Identifier',
                    $filter->getIdentifiers()
                )
            );
        }

        return $query;
    }

    /**
     *  Deletes records older than 7 days.
     *
     * @return void
     */
    public function deleteOldRecords()
    {
        $updatedAt = new \DateTime();
        $updatedAt->setTime(0, 0, 0);
        $updatedAt->modify('-7 days');

        $query = $this->createQueryBuilder($this->getTableAlias());
        $query
            ->select()
            ->where('article.updatedAt <= :updated_at')
            ->setParameter('updated_at', $updatedAt)
            ->orWhere($query->expr()->isNull('article.updatedAt'));

        foreach ($query->getQuery()->execute() as $article) {
            $this->remove($article);
        }

        $this->persistenceManager->persistAll();
    }

    /**
     * Returns orphan objects (e.g. they are used nowhere and can safely be deleted
     * without any consequences).
     *
     * @return iterable
     */
    public function findOrphans()
    {
        $query = $this->createQueryBuilder('article');
        $query->where($query->expr()->isNull('article.client'));
        return $query->getQuery()->execute();
    }
}
