<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Repository;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\QueryBuilder;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Persistence\Doctrine\Repository;
use Neos\Flow\Persistence\QueryResultInterface;
use Newland\Toubiz\Sync\Neos\Domain\Filter\FilterInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\AbstractEntity;
use Newland\Toubiz\Sync\Neos\Service\ArticleClientFilterService;

/**
 * Abstract repository.
 *
 * @Flow\Scope("singleton")
 *
 * @method AbstractEntity findOneByOriginalId(string $originalId)
 */
abstract class AbstractRepository extends Repository
{

    /**
     * @var ArticleClientFilterService
     * @Flow\Inject
     */
    protected $articleClientFilterService;

    /**
     * @param FilterInterface $filter
     * @param QueryBuilder $query
     * @return QueryBuilder
     */
    abstract protected function applyFilter(FilterInterface $filter, QueryBuilder $query): QueryBuilder;

    /**
     * Counts items and pages for given filter.
     *
     * @param FilterInterface $filter
     * @param int $perPage
     * @return array
     */
    public function countByFilter(FilterInterface $filter, $perPage)
    {
        $query = $this->queryForFilter($filter);

        // Reset pagination parameters.
        $query->setMaxResults(null);
        $query->setFirstResult(null);

        $query->select('COUNT(' . $this->getTableAlias() . ') as total');
        $result = $query->getQuery()->execute()[0];
        $pages = floor($result['total'] / $perPage) + ($result['total'] % $perPage == 0 ? 0 : 1);
        return [
            'items' => $result['total'],
            'pages' => ($pages == 0 ? 1 : $pages),
        ];
    }

    /**
     * Find entities matching given filter.
     *
     * @param FilterInterface $filter
     * @return QueryResultInterface
     */
    public function findByFilter(FilterInterface $filter)
    {
        return $this->queryForFilter($filter)
            ->getQuery()
            ->execute();
    }

    public function queryForFilter(FilterInterface $filter): QueryBuilder
    {
        return $this->applyFilter(
            $filter,
            $this->createQueryBuilder($this->getTableAlias())
        );
    }

    /**
     * Removes all records from given array of ids.
     *
     * @param array $ids
     * @return void
     */
    public function removeByIds(array $ids)
    {
        $query = $this->createQueryBuilder($this->getTableAlias());
        $query->delete()
            ->where($this->getTableAlias() . '.Persistence_Object_Identifier IN(:ids)')
            ->setParameter('ids', $ids);

        $query->getQuery()->execute();
    }

    /**
     * Applies basic filter functions (from the abstract filter) onto
     * the given query builder.
     *
     * @param FilterInterface $filter
     * @param QueryBuilder $query
     * @return void
     */
    protected function applyBasicFilter(FilterInterface $filter, QueryBuilder $query)
    {
        if ($filter->hasLeftJoins()) {
            foreach ($filter->getLeftJoins() as $field => $table) {
                $query->leftJoin($field, $table);
            }
            $query->distinct();
        }
        if ($filter->hasInnerJoins()) {
            foreach ($filter->getInnerJoins() as $field => $table) {
                $query->innerJoin($field, $table);
            }
            $query->distinct();
        }

        if ($filter->hasOrderBy()) {
            foreach ($filter->getOrderBy() as $field => $order) {
                $query->addOrderBy($field, $order);
            }
        }

        if ($filter->hasGroupBy()) {
            foreach ($filter->getGroupBy() as $field) {
                $query->addGroupBy($field);
            }
        }

        if ($filter->hasLimit()) {
            $query->setMaxResults($filter->getLimit());
        }

        if ($filter->hasOffset()) {
            $query->setFirstResult($filter->getOffset());
        }

        if ($filter->hasExcludes()) {
            $excludes = [];
            foreach ($filter->getExcludes() as $exclude) {
                $excludes[] = $exclude->getOriginalId();
            }

            $query->andWhere(
                $query->expr()->notIn(
                    $this->getTableAlias() . '.originalId',
                    $excludes
                )
            );
        }
    }

    /**
     * Helper for getting the table alias used inside queries.
     *
     * This may be a temporary solution!
     *
     * @return string
     */
    protected function getTableAlias()
    {
        $parts = explode('\\', get_class($this));
        $name = array_values(array_slice($parts, -1))[0];
        return lcfirst(str_replace('Repository', '', $name));
    }
}
