<?php
namespace Newland\Toubiz\Sync\Neos\Command;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\Service\ServiceFactory;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DbService\PointOfInterestAdapter;
use Newland\Toubiz\Sync\Neos\Behaviour\EventEmitter;
use Newland\Toubiz\Sync\Neos\Importer\ArticleImporter;

/**
 * Articles command controller.
 *
 * Provides commands to manipulate article data.
 *
 * @Flow\Scope("singleton")
 */
class ArticlesCommandController extends AbstractCommandController
{
    use EventEmitter;

    const EVENT_START = 'start';
    const EVENT_PROGRESS = 'progress';
    const EVENT_END = 'end';

    /**
     * Synchronize command.
     *
     * Updates local articles database from API data source.
     *
     * @param bool $quiet
     * @return void
     * @throws \Exception
     */
    public function synchronizeCommand($quiet = false)
    {
        if (!$quiet) {
            $this->on(static::EVENT_START, function (string $type) {
                $this->outputLine(PHP_EOL . 'Synchronizing ' . $type);
                $this->output->progressStart();
            });
            $this->on(static::EVENT_PROGRESS, function () {
                $this->output->progressAdvance();
            });
            $this->on(static::EVENT_END, function () {
                $this->output->progressFinish();
            });
        }

        $configuration = $this->getConfigurationForService('Tportal/Api');
        if ($this->hasClients($configuration)) {
            foreach ($this->getClients($configuration) as $clientConfiguration) {
                $this->synchronizeLodgingsFromTportal($clientConfiguration);
            }
        }

        $configuration = $this->getConfigurationForService('Toubiz/Legacy/Db');
        if ($configuration) {
            $this->synchronizePointOfInterestsFromDbService($configuration);
        }

        $configuration = $this->getConfigurationForService('Toubiz/Legacy/GastronomyApi');
        if ($configuration) {
            $this->synchronizeGastronomyFromApi($configuration);
        }

        $configuration = $this->getConfigurationForService('Toubiz/Legacy/DirectMarketerApi');
        if ($configuration) {
            $this->synchronizeDirectMarketersFromApi($configuration);
        }

        $configuration = $this->getConfigurationForService('Outdooractive/Api');
        if ($configuration) {
            $this->synchronizeToursFromApi($configuration);
        }
        $importer = new ArticleImporter;
        $importer->deleteOldRecords();
    }

    /**
     * Synchronizes lodging data as articles from TPortal.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizeLodgingsFromTportal($configuration)
    {
        $this->emit(static::EVENT_START, 'Lodgings');
        $client = $configuration['client'];

        /** @var \Newland\Toubiz\Api\Service\Tportal\ApiService $service */
        $service = ServiceFactory::get('Tportal/Api');
        $service->setClientName($client);

        $service->fetch(
            'lodgings',
            function ($record) use ($client) {
                $this->emit(static::EVENT_PROGRESS);
                $importer = new ArticleImporter;
                $importer->setClient($client);
                $importer->import($record);
            }
        );

        $this->emit(static::EVENT_END, 'Lodgings');
    }

    /**
     * Synchronizes POI data from legacy toubiz DB service.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizePointOfInterestsFromDbService($configuration)
    {
        $this->emit(static::EVENT_START, 'POI');

        $service = ServiceFactory::get('Toubiz/Legacy/Db');
        $service->setClientName($configuration['client']);
        $service->setApiKey($configuration['apiKey']);

        $minLevelOfMaintenance = $configuration['minLevelOfMaintenance'] ?? 0;

        $service->fetch('pointOfInterests', function (PointOfInterestAdapter $record) use ($minLevelOfMaintenance) {
            $this->emit(static::EVENT_PROGRESS);
            if ($record->getLevelOfMaintenance() > $minLevelOfMaintenance) {
                (new ArticleImporter())->import($record);
            } else {
                (new ArticleImporter())->remove($record);
            }
        });

        $this->emit(static::EVENT_END, 'POI');
    }

    /**
     * Synchronizes gastronomy data from legacy API.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizeGastronomyFromApi($configuration)
    {
        $this->emit(static::EVENT_START, 'Gastronomy');

        $service = ServiceFactory::get('Toubiz/Legacy/GastronomyApi');
        $service->setApiKey($configuration['apiKey']);

        $service->fetch(
            'gastronomy',
            function ($record) {
                $this->emit(static::EVENT_PROGRESS);
                $importer = new ArticleImporter;
                $importer->import($record);
            }
        );

        $this->emit(static::EVENT_END, 'Gastronomy');
    }

    /**
     * Synchronizes direct marketers from toubiz legacy API.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizeDirectMarketersFromApi(array $configuration)
    {
        $this->emit(static::EVENT_START, 'Direct Marketers');

        $service = ServiceFactory::get('Toubiz/Legacy/DirectMarketerApi');
        $service->setApiKey($configuration['apiKey']);

        $service->fetch(
            'directMarketers',
            function ($record) {
                $this->emit(static::EVENT_PROGRESS);
                $importer = new ArticleImporter;
                $importer->setOpeningTimesFormat(ArticleConstants::OPENING_TIMES_FORMAT_LEGACY);
                $importer->import($record);
            }
        );

        $this->emit(static::EVENT_END, 'Direct Marketers');
    }

    /**
     * Synchronizes tours.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizeToursFromApi($configuration)
    {
        $this->emit(static::EVENT_START, 'Tours');

        $service = ServiceFactory::get('Outdooractive/Api');
        $service->setClientName($configuration['client']);
        $service->setApiKey($configuration['apiKey']);

        $records = $service->fetchTours();
        if (!$records) {
            throw new \Exception('No tour data received for synchronizing!');
        }

        foreach ($records as $record) {
            $this->emit(static::EVENT_PROGRESS);
            $importer = new ArticleImporter;
            $importer->import($record);
        }

        $this->emit(static::EVENT_END, 'Tours');
    }
}
