<?php declare(strict_types=1);

namespace Newland\Toubiz\Sync\Neos\Tests\Unit\Domain\Repository;

use Doctrine\Common\Collections\ArrayCollection;
use Neos\Flow\Tests\FunctionalTestCase;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Model\EventDate;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventRepository;
use Newland\Toubiz\Sync\Neos\Tests\Factory\EventDateFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\EventFactory;

class EventRepositoryTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;

    /** @var EventFactory */
    protected $factory;

    /** @var EventDateFactory */
    protected $dateFactory;

    /** @var EventRepository */
    protected $subject;

    public function setUp(): void
    {
        parent::setUp();
        $this->factory = new EventFactory($this->objectManager);
        $this->dateFactory = new EventDateFactory($this->objectManager);
        $this->subject = $this->objectManager->get(EventRepository::class);
    }

    public function testFindAllForDataSourceRespectsLanguage(): void
    {
        $french = $this->factory->create([ 'language' => 'fr' ]);
        $this->dateFactory->create([ 'language' => 'fr', 'event' => $french ]);
        $spanish = $this->factory->create([ 'language' => 'es' ]);
        $this->dateFactory->create([ 'language' => 'es', 'event' => $spanish ]);

        $this->subject->setLanguage('fr');
        $result = $this->subject->findAllForDataSource(new \DateTime('1970-01-01T00:00:00'));
        $this->assertCount(1, $result, 'Only one result should be found');
        $this->assertEquals($french->getPersistenceObjectIdentifier(), $result[0]['value'], 'Should find french');
        $this->assertNotEquals($spanish->getPersistenceObjectIdentifier(), $result[0]['value'], 'Should not find spanish');
    }

    public function testCanFindLanguageVariants(): void
    {
        $de = (new EventFactory($this->objectManager))->create([ 'language' => 'de', 'client' => 'foo', 'languageGrouping' => 'foo-bar' ]);
        $en = (new EventFactory($this->objectManager))->create([ 'language' => 'en', 'client' => 'foo', 'languageGrouping' => 'foo-bar' ]);
        $fr = (new EventFactory($this->objectManager))->create([ 'language' => 'fr', 'client' => 'foo', 'languageGrouping' => 'foo-bar' ]);

        $variants = $this->subject->findLanguageVariants($de);

        $this->assertArrayHasKey('de', $variants);
        $this->assertArrayHasKey('en', $variants);
        $this->assertArrayHasKey('fr', $variants);

        $this->assertEquals($de->getPersistenceObjectIdentifier(), $variants['de']->getPersistenceObjectIdentifier());
        $this->assertEquals($en->getPersistenceObjectIdentifier(), $variants['en']->getPersistenceObjectIdentifier());
        $this->assertEquals($fr->getPersistenceObjectIdentifier(), $variants['fr']->getPersistenceObjectIdentifier());
    }


    public function testDoesNotFindEmptyLanguageAsVariant(): void
    {
        $de = (new EventFactory($this->objectManager))->create([ 'language' => 'de', 'client' => 'foo', 'languageGrouping' => 'foo-bar' ]);
        $en = (new EventFactory($this->objectManager))->create([ 'language' => 'en', 'client' => 'foo', 'languageGrouping' => 'foo-bar' ]);
        $empty = (new EventFactory($this->objectManager))->create([ 'language' => null, 'client' => 'foo', 'languageGrouping' => 'foo-bar' ]);

        $variants = $this->subject->findLanguageVariants($de);
        $this->assertCount(2, $variants);
        $this->assertArrayHasKey('de', $variants);
        $this->assertArrayHasKey('en', $variants);
    }
}
