<?php declare(strict_types=1);
namespace Newland\Toubiz\Sync\Neos\Domain\Model;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\ResourceManagement\PersistentResource;
use Neos\Flow\ResourceManagement\ResourceManager;
use Newland\Toubiz\Api\ObjectAdapter\Concern\MediumConstants;

/**
 * A multimedia item.
 *
 * @Flow\Entity
 * @ORM\Table(indexes={
 *      @ORM\Index(name="newland_toubiz_sync_neos_medium_import_ident", columns={"originalid"})
 * })
 */
class Medium extends AbstractEntity
{
    /**
     * @ORM\ManyToMany(
     *     targetEntity="\Newland\Toubiz\Sync\Neos\Domain\Model\Article",
     *     mappedBy="media",
     *     fetch="LAZY"
     * )
     * @var Collection<int, Article>
     */
    protected $articles;

    /**
     * @ORM\ManyToMany(
     *     targetEntity="\Newland\Toubiz\Sync\Neos\Domain\Model\Event",
     *     mappedBy="media",
     *     fetch="LAZY"
     * )
     * @var Collection<int, Event>
     */
    protected $events;

    /**
     * @ORM\ManyToMany(
     *     targetEntity="\Newland\Toubiz\Sync\Neos\Domain\Model\Service",
     *     mappedBy="media",
     *     fetch="LAZY"
     * )
     * @var Collection<int, Service>
     */
    protected $services;

    /**
     * @var string|null
     * @ORM\Column(type="text", nullable=true)
     */
    protected $title;

    /**
     * @var string
     */
    protected $type;

    /**
     * @var string
     * @ORM\Column(type="text")
     */
    protected $sourceUri = '#';

    /**
     * @var string|null
     * @ORM\Column(type="text", nullable=true)
     */
    protected $previewUri = '#';

    /**
     * @var string|null
     * @ORM\Column(type="text", nullable=true)
     */
    protected $copyright;

    /**
     * @var PersistentResource|null
     * @ORM\OneToOne(orphanRemoval=true, cascade={"all"})
     */
    protected $resource;

    /**
     * @var ResourceManager
     * @Flow\Inject()
     */
    protected $resourceManager;

    public function __construct()
    {
        $this->articles = new ArrayCollection();
        $this->events = new ArrayCollection();
    }

    public function setTitle(?string $title): void
    {
        $this->title = $title;
    }

    public function getTitle(): ?string
    {
        return $this->title;
    }

    public function setType(?string $type): void
    {
        $this->type = $type;
    }

    public function getType(): ?string
    {
        return $this->type;
    }

    public function setSourceUri(string $sourceUri): void
    {
        $this->sourceUri = $sourceUri;
    }

    public function getSourceUri(): string
    {
        return $this->sourceUri;
    }

    public function setPreviewUri(?string $previewUri): void
    {
        $this->previewUri = $previewUri;
    }

    public function getPreviewUri(): ?string
    {
        return $this->previewUri;
    }

    public function setCopyright(?string $copyright): void
    {
        $this->copyright = $copyright;
    }

    public function getCopyright(): ?string
    {
        return $this->copyright;
    }

    public function getResource(): ?PersistentResource
    {
        return $this->resource;
    }

    public function setResource(?PersistentResource $resource): void
    {
        $this->resource = $resource;
    }

    public function getUri(): string
    {
        return $this->getResourceUri() ?? $this->sourceUri;
    }

    public function getIsVideo(): bool
    {
        return $this->type === MediumConstants::TYPE_VIDEO;
    }

    public function resourceExistsOnDisk(): bool
    {
        if (!$this->resource) {
            return false;
        }

        // If the file on disk corresponding to the persistent resource does not exist, then
        // this method returns `false`.
        if ($this->resource->getStream() === false) {
            return false;
        }

        return true;
    }

    private function getResourceUri(): ?string
    {
        if (!$this->resource || !$this->resourceExistsOnDisk()) {
            return null;
        }

        $uri = $this->resourceManager->getPublicPersistentResourceUri($this->resource);
        if (!$uri || !\is_string($uri)) {
            return null;
        }
        return $uri;
    }
}
