<?php
namespace Newland\Toubiz\Sync\Neos\Importer;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;

/**
 * Article importer.
 *
 * @Flow\Scope("singleton")
 */
class ArticleImporter extends AbstractImporter
{
    /**
     * @var ArticleAdapterInterface
     */
    protected $data;

    /**
     * @var Article
     */
    protected $article;

    /**
     * @Flow\Inject
     * @var ArticleRepository
     */
    protected $articleRepository;

    /**
     * Import method.
     *
     * Persist given data by creating new objects or updating existing ones.
     *
     * @param ArticleAdapterInterface $data
     * @return Article
     */
    public function import($data)
    {
        $this->data = $data;
        $this->article = $this->articleRepository->findOneByOriginalId($this->data->getExternalId());
        $persisted = !!$this->article;

        if (!$this->article) {
            $this->article = new Article;
        }

        $this->mapSimpleValues();

        if ($persisted) {
            $this->articleRepository->update($this->article);
        } else {
            $this->articleRepository->add($this->article);
        }

        $this->mapAddresses();
        $this->mapCategories();
        $this->mapFiles();
        $this->mapMedia();

        $this->articleRepository->update($this->article);
        $this->persistenceManager->persistAll();
    }

    /**
     * Maps simple values onto the article.
     *
     * @return void
     */
    protected function mapSimpleValues()
    {
        if (empty($this->article->getOriginalId())) {
            $this->article->setOriginalId($this->data->getExternalId());
        }

        $this->article->setMainType($this->data->getMainType());
        $this->article->setName($this->data->getName());
        $this->article->setAbstract($this->data->getAbstract());
        $this->article->setDescription($this->data->getDescription());
        $this->article->setStreet($this->data->getStreet());
        $this->article->setZip($this->data->getZip());
        $this->article->setCity($this->data->getCity());
        $this->article->setCountry($this->data->getCountry());
        $this->article->setLatitude($this->data->getLatitude());
        $this->article->setLongitude($this->data->getLongitude());
        $this->article->setAttributes($this->data->getAttributes());
        $this->article->setFacebookUri($this->data->getFacebookUri());
        $this->article->setTwitterUri($this->data->getTwitterUri());
        $this->article->setInstagramUri($this->data->getInstagramUri());
        $this->article->setYoutubeUri($this->data->getYoutubeUri());
        $this->article->setSourceName($this->data->getSourceName());
        $this->article->setAuthorName($this->data->getAuthorName());
        $this->article->setBookingUri($this->data->getBookingUri());
        $this->article->setDetailUri($this->data->getDetailUri());
    }

    /**
     * Maps addresses onto the article.
     *
     * @return void
     */
    protected function mapAddresses()
    {
        if (empty($this->data->getAddresses())) {
            return;
        }

        $addresses = $this->article->getAddresses();
        $addresses->clear();
        foreach ($this->data->getAddresses() as $entry) {
            $importer = new AddressImporter;
            $addresses->add($importer->import($entry));
        }
        $this->article->setAddresses($addresses);
    }

    /**
     * Maps categories onto the article.
     *
     * @return void
     */
    protected function mapCategories()
    {
        if (empty($this->data->getCategories())) {
            return;
        }

        $categories = $this->article->getCategories();
        $categories->clear();
        foreach ($this->data->getCategories() as $entry) {
            $importer = new CategoryImporter;
            $category = $importer->import($entry);
            if (!$categories->contains($category)) {
                $categories->add($category);
            }
        }
        $this->article->setCategories($categories);
    }

    /**
     * Maps multimedia items onto the article.
     *
     * @return void
     */
    public function mapMedia()
    {
        if (empty($this->data->getMedia())) {
            return;
        }

        $media = $this->article->getMedia();
        $media->clear();
        foreach ($this->data->getMedia() as $entry) {
            $importer = new MediumImporter;
            $medium = $importer->import($entry);
            if (!$media->contains($medium)) {
                $media->add($medium);
            }
        }
        $this->article->setMedia($media);
    }

    /**
     * Maps files onto the article.
     *
     * @return void
     */
    public function mapFiles()
    {
        if (empty($this->data->getFiles())) {
            return;
        }

        $files = $this->article->getFiles();
        $files->clear();
        foreach ($this->data->getFiles() as $entry) {
            $importer = new FileImporter;
            $file = $importer->import($entry);
            if (!$files->contains($file)) {
                $files->add($file);
            }
        }
        $this->article->setFiles($files);
    }
}
