<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Repository;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\QueryBuilder;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Persistence\Generic\Query;
use Neos\Flow\Persistence\QueryResultInterface;
use Newland\Toubiz\Sync\Neos\Domain\Filter\ArticleFilter;

/**
 * Article repository.
 *
 * @Flow\Scope("singleton")
 */
class ArticleRepository extends AbstractRepository
{
    const MIN_LATITUDE = -90;
    const MAX_LATITUDE = 90;
    const MIN_LONGITUDE = -180;
    const MAX_LONGITUDE = 180;

    /**
     * Find articles matching given filter.
     *
     * TODO refactor into abstract
     *
     * @param ArticleFilter $filter
     * @return QueryResultInterface
     */
    public function findByFilter(ArticleFilter $filter)
    {
        return $this->queryForFilter($filter)->getQuery()->execute();
    }

    /**
     * Specific method for fetching all data for a data source
     * (e.g. multi-select in a neos node type property) in the
     * most performant way.
     *
     * @return array
     */
    public function findAllForDataSource()
    {
        $query = $this->createQueryBuilder('article');
        $query->select([
            'article.name AS article_name',
            'article.Persistence_Object_Identifier AS article_identifier',
            'category.title AS category_title'

        ])->join('article.categories', 'category')
          ->addOrderBy('category.title', 'asc')
          ->addOrderBy('article.name', 'asc');
        return $query->getQuery()->getScalarResult();
    }

    public function queryForFilter(ArticleFilter $filter): QueryBuilder
    {
        return $this->applyFilter(
            $filter,
            $this->createQueryBuilder('article')
        );
    }

    /**
     * Applies filter functions (from the article filter) onto
     * the given query builder.
     *
     * @param ArticleFilter $filter
     * @param QueryBuilder $query
     * @return void
     */
    protected function applyFilter(ArticleFilter $filter, QueryBuilder $query): QueryBuilder
    {
        parent::applyBasicFilter($filter, $query);

        // Category is always joined as e.g. sorting relies on it.
        $query->leftJoin('article.categories', 'category');

        if ($filter->hasMainType()) {
            $query->andWhere(
                $query->expr()->eq('article.mainType', $filter->getMainType())
            );
        }
        if ($filter->hasCategories()) {
            $query->andWhere(
                $query->expr()->in(
                    'category.' . $filter->getCategoriesIdentifierField(),
                    $filter->getCategories()
                )
            );
        }
        if ($filter->hasLatitude()) {
            $query->andWhere(
                $query->expr()->like(
                    'article.latitude',
                    $query->expr()->literal($filter->getLatitude() . '%')
                )
            );
        }
        if ($filter->hasLongitude()) {
            $query->andWhere(
                $query->expr()->like(
                    'article.longitude',
                    $query->expr()->literal($filter->getLongitude() . '%')
                )
            );
        }
        if ($filter->hasZips()) {
            $query->andWhere(
                $query->expr()->in(
                    'article.zip',
                    $filter->getZips()
                )
            );
        }

        if ($filter->getExcludeUnsafeCoordinates()) {
            $query->andWhere(
                $query->expr()->isNotNull('article.latitude'),
                $query->expr()->isNotNull('article.longitude'),
                $query->expr()->lte('article.latitude', static::MAX_LATITUDE),
                $query->expr()->gte('article.latitude', static::MIN_LATITUDE),
                $query->expr()->lte('article.longitude', static::MAX_LONGITUDE),
                $query->expr()->gte('article.longitude', static::MIN_LONGITUDE),
                $query->expr()->neq('article.latitude', 0),
                $query->expr()->neq('article.longitude', 0)
            );
        }

        if ($filter->hasIdentifiers()) {
            $query->andWhere(
                $query->expr()->in(
                    'article.Persistence_Object_Identifier',
                    $filter->getIdentifiers()
                )
            );
        }

        return $query;
    }
}
