<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Filter;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\PersistentCollection;

/**
 * Article filter class.
 */
class ArticleFilter extends AbstractFilter implements FilterInterface
{
    /**
     * @var array
     */
    protected $categories = [];

    /**
     * Field to match for categories.
     *
     * Default ist persistence_object_identifier, but e.g. the
     * category mapping returns originalId values.
     *
     * @var string
     */
    protected $categoriesIdentifierField = 'Persistence_Object_Identifier';

    /**
     * @var float
     */
    protected $latitude;

    /**
     * @var float
     */
    protected $longitude;

    /**
     * @var int
     */
    protected $mainType;

    /**
     * Precision for lat/long comparison.
     * - 1 => ~11.1km
     * - 2 => ~1.1km
     * - 3 => ~110m
     * - 4 => ~11m
     *
     * @var integer
     */
    protected $coordinatePrecision;

    /**
     * @var array
     */
    protected $zips;

    /**
     * @var bool
     */
    protected $excludeUnsafeCoordinates = false;

    /**
     * @var string[]
     */
    protected $identifiers = [];

    /**
     * Initializes filter options.
     *
     * @param array $config
     * @return self
     */
    public function initialize(array $config)
    {
        parent::initialize($config);
        return $this;
    }

    /**
     * @return bool
     */
    public function hasCategories()
    {
        return (count($this->categories) > 0);
    }

    /**
     * @param array $categories
     * @return self
     */
    public function setCategories(array $categories)
    {
        $this->categories = $categories;
        return $this;
    }

    /**
     * @param PersistentCollection $categories
     * @return self
     */
    public function setCategoryCollection(PersistentCollection $categories)
    {
        $this->categories = [];
        foreach ($categories as $category) {
            $this->categories[] = $category->getPersistenceObjectIdentifier();
        }
        return $this;
    }

    /**
     * @return array
     */
    public function getCategories()
    {
        return $this->categories;
    }

    /**
     * @param string $fieldName
     * @return self
     */
    public function setCategoriesIdentifierField($fieldName)
    {
        return $this->categoriesIdentifierField = $fieldName;
    }

    /**
     * @return string
     */
    public function getCategoriesIdentifierField()
    {
        return $this->categoriesIdentifierField;
    }

    /**
     * @param float|null $latitude
     * @return self
     */
    public function setLatitude($latitude)
    {
        $this->latitude = $latitude;
        return $this;
    }

    /**
     * @return float|null
     */
    public function getLatitude()
    {
        if (!empty($this->coordinatePrecision)) {
            return bcdiv($this->latitude, 1, $this->coordinatePrecision);
        }
        return $this->latitude;
    }

    /**
     * @return bool
     */
    public function hasLatitude()
    {
        return !empty($this->latitude);
    }

    /**
     * @param float|null $longitude
     * @return self
     */
    public function setLongitude($longitude)
    {
        $this->longitude = $longitude;
        return $this;
    }

    /**
     * @return float|null
     */
    public function getLongitude()
    {
        if (!empty($this->coordinatePrecision)) {
            return bcdiv($this->longitude, 1, $this->coordinatePrecision);
        }
        return $this->longitude;
    }

    /**
     * @return bool
     */
    public function hasLongitude()
    {
        return !empty($this->longitude);
    }

    /**
     * @var integer $precision
     * @return self
     */
    public function setCoordinatePrecision($precision)
    {
        $this->coordinatePrecision = $precision;
        return $this;
    }

    /**
     * @return int
     */
    public function getCoordinatePrecision()
    {
        return $this->coordinatePrecision;
    }

    /**
     * @param int $mainType
     * @return void
     */
    public function setMainType($mainType)
    {
        $this->mainType = $mainType;
    }

    /**
     * @return int|null
     */
    public function getMainType()
    {
        return $this->mainType;
    }

    /**
     * @return bool
     */
    public function hasMainType()
    {
        return !is_null($this->mainType);
    }

    public function hasZips(): bool
    {
        return $this->zips && \count($this->zips) > 0;
    }

    /**
     * @return array
     */
    public function getZips()
    {
        return $this->zips;
    }

    /**
     * @param array $zips
     */
    public function setZips(array $zips)
    {
        $this->zips = $zips;
    }

    /**
     * @return bool
     */
    public function getExcludeUnsafeCoordinates(): bool
    {
        return $this->excludeUnsafeCoordinates;
    }

    /**
     * @param bool $excludeUnsafeCoordinates
     */
    public function setExcludeUnsafeCoordinates(bool $excludeUnsafeCoordinates)
    {
        $this->excludeUnsafeCoordinates = $excludeUnsafeCoordinates;
    }


    public function setIdentifiers(array $identifiers)
    {
        $this->identifiers = $identifiers;
    }

    public function getIdentifiers(): array
    {
        return$this->identifiers;
    }

    public function hasIdentifiers(): bool
    {
        return !empty($this->identifiers);
    }
}
