<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Repository;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Persistence\QueryInterface;
use Neos\Flow\Persistence\Doctrine\QueryResult;
use Newland\Toubiz\Api\ObjectAdapter\Concern\SlopeConstants;
use Newland\Toubiz\Sync\Neos\Domain\Filter\SlopeFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Region;

/**
 * Slope repository.
 *
 * @Flow\Scope("singleton")
 */
class SlopeRepository extends AbstractRepository
{
    /**
     * Find slopes matching given filter.
     *
     * @param SlopeFilter $filter
     * @return QueryResultInterface
     */
    public function findByFilter(SlopeFilter $filter)
    {
        $query = $this->createQueryBuilder('slope');
        $this->applyBasicFilter($filter, $query);
        $query->leftJoin('slope.regions', 'region');

        if ($filter->hasRegions()) {
            $query->andWhere(
                $query->expr()->in('region', $filter->getRegions())
            );
        }
        if ($filter->hasType()) {
            $query->andWhere(
                $query->expr()->in('slope.type', $filter->getTypes())
            );
        }

        return $query->getQuery()->execute();
    }

    /**
     * Calculates statistics.
     *
     * @param Region $region
     * @return array
     */
    public function getStatistics(array $regions = [])
    {
        $query = $this->createQueryBuilder('slope');
        $query->leftJoin('slope.regions', 'region');

        $constraint = '';
        if (!empty($regions)) {
            $regionIds = [];
            foreach ($regions as $region) {
                $regionIds[] = $region->getPersistenceObjectIdentifier();
            }
            $query->setParameter('regionIds', $regionIds);
            $constraint = 'AND region IN(:regionIds)';
        }

        $query->select('
            SUM (
                CASE WHEN
                    slope.status = ' . SlopeConstants::STATUS_OPEN . '
                    AND slope.type = ' . SlopeConstants::TYPE_SKI . '
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS ski_open,
            SUM (
                CASE WHEN
                    slope.type = ' . SlopeConstants::TYPE_SKI . '
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS ski_total,

            SUM (
                CASE WHEN
                    slope.status = ' . SlopeConstants::STATUS_OPEN . '
                    AND slope.type = ' . SlopeConstants::TYPE_HIKING_TRAIL . '
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS hike_open,
            SUM (
                CASE WHEN
                    slope.type = ' . SlopeConstants::TYPE_HIKING_TRAIL . '
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS hike_total,

            SUM (
                CASE WHEN
                    slope.type IN(
                        ' . SlopeConstants::TYPE_CROSS_COUNTRY_SKIING . ',
                        ' . SlopeConstants::TYPE_CROSS_COUNTRY_SKIING_SKATING . '
                    )
                    AND slope.status = ' . SlopeConstants::STATUS_OPEN . '
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS cross_country_skiing_open,
            SUM (
                CASE WHEN
                    slope.type IN(
                        ' . SlopeConstants::TYPE_CROSS_COUNTRY_SKIING . ',
                        ' . SlopeConstants::TYPE_CROSS_COUNTRY_SKIING_SKATING . '
                    )
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS cross_country_skiing_total
        ');
        return $query->getQuery()->execute()[0];
    }
}
