<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Filter;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

/**
 * Event date filter class.
 *
 * Contrary to the event filter, filtering based on specific date values
 * is more accurate, and provides better usability in cases where you
 * want to list upcoming event dates rather than events.
 */
class EventDateFilter extends AbstractFilter implements FilterInterface
{
    /**
     * @var string
     */
    const DATE_FORMAT = 'Y-m-d';

    /**
     * @var array
     */
    protected $categories = [];

    /**
     * Queries the from-date as minimum value.
     *
     * @var \DateTime|null
     */
    protected $fromDate;

    /**
     * Queries the from-date as maximum value.
     * @var \DateTime|null
     */
    protected $fromMaxDate;

    /**
     * @var \DateTime|null
     */
    protected $toDate;

    /**
     * @var bool|null
     */
    protected $highlight;

    /**
     * Initializes filter options.
     *
     * @param array $config
     * @return EventDateFilter
     */
    public function initialize(array $config)
    {
        parent::initialize($config);

        if (array_key_exists('categories', $config)) {
            $this->setCategories($config['categories']);
        }
        if (array_key_exists('fromDate', $config)) {
            $this->setFromDate($this->asDateTime($config['fromDate']));
        } else {
            // ignore past events
            $this->setFromDate($this->asDateTime('now'));
        }
        if (array_key_exists('toDate', $config)) {
            $this->setToDate($this->asDateTime($config['toDate']));
        }

        return $this;
    }

    /**
     * @return bool
     */
    public function hasFromDate()
    {
        return !is_null($this->fromDate);
    }

    /**
     * @return \DateTime|null
     */
    public function getFromDate()
    {
        return $this->fromDate;
    }

    /**
     * @param \DateTime $date
     * @return self
     */
    public function setFromDate(\DateTime $date)
    {
        $this->fromDate = $date;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasToDate()
    {
        return !is_null($this->toDate);
    }

    /**
     * @return \DateTime|null
     */
    public function getToDate()
    {
        return $this->toDate;
    }

    /**
     * @param \DateTime $date
     * @return self
     */
    public function setToDate(\DateTime $date)
    {
        $this->toDate = $date;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasHighlight()
    {
        return !is_null($this->highlight);
    }

    /**
     * @return bool|null
     */
    public function getHighlight()
    {
        return $this->highlight;
    }

    /**
     * @param bool|null $highlight
     * @return self
     */
    public function setHighlight($highlight)
    {
        $this->highlight = $highlight;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasFromMaxDate()
    {
        return !is_null($this->fromMaxDate);
    }

    /**
     * @return \DateTime|null
     */
    public function getFromMaxDate()
    {
        return $this->fromMaxDate;
    }

    /**
     * @param \DateTime|null $date
     * @return self
     */
    public function setFromMaxDate(\DateTime $date)
    {
        $this->fromMaxDate = $date;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasCategories()
    {
        return !empty($this->categories);
    }

    /**
     * @return string[]
     */
    public function getCategories()
    {
        return $this->categories;
    }

    /**
     * @param string[]
     * @return self
     */
    public function setCategories(array $categories)
    {
        $this->categories = $categories;
        return $this;
    }
}
