<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Filter;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

/**
 * Abstract filter class.
 */
abstract class AbstractFilter
{
    /**
     * @var int
     */
    const JOIN_LEFT = 0;

    /**
     * @var int
     */
    const JOIN_INNER = 1;

    /**
     * @var int
     */
    protected $limit;

    /**
     * @var int
     */
    protected $offset;

    /**
     * @var array
     */
    protected $orderBy = [];

    /**
     * @var array
     */
    protected $groupBy = [];

    /**
     * @var array Objects to exclude from the query result.
     */
    protected $excludes = [];

    /**
     * @var array Joins that are needed for filtering.
     */
    protected $joins = [
        self::JOIN_LEFT => [],
        self::JOIN_INNER => [],
    ];

    /**
     * Initializes filter options.
     *
     * @param array $config
     * @return self
     */
    public function initialize(array $filter)
    {
        if (array_key_exists('limit', $filter)) {
            $this->limit = $filter['limit'];
        }
        if (array_key_exists('offset', $filter)) {
            $this->offset = $filter['offset'];
        }

        return $this;
    }

    /**
     * @return bool
     */
    public function hasLimit()
    {
        return !is_null($this->limit);
    }

    /**
     * @return int|null
     */
    public function getLimit()
    {
        return $this->limit;
    }

    /**
     * @param int|null $limit
     * @return self
     */
    public function setLimit($limit)
    {
        $this->limit = $limit;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasOffset()
    {
        return !is_null($this->offset);
    }

    /**
     * @return int|null
     */
    public function getOffset()
    {
        return $this->offset;
    }

    /**
     * @param int|null $offset
     * @return self
     */
    public function setOffset($offset)
    {
        $this->offset = $offset;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasOrderBy()
    {
        return !empty($this->orderBy);
    }

    /**
     * @return array|null
     */
    public function getOrderBy()
    {
        return $this->orderBy;
    }

    /**
     * @param array|null $orderBy
     * @return self
     */
    public function setOrderBy($orderBy)
    {
        $this->orderBy = $orderBy;
        return $this;
    }

    /**
     * @param array $excludes
     * @return self
     */
    public function setExcludes(array $excludes)
    {
        $this->excludes = $excludes;
        return $this;
    }

    /**
     * @return array
     */
    public function getExcludes()
    {
        return $this->excludes;
    }

    /**
     * @param mixed $exclude
     * @return self
     */
    public function addExclude($exclude)
    {
        $this->excludes[] = $exclude;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasExcludes()
    {
        return !empty($this->excludes);
    }

    /**
     * @param string $field
     * @param string $table
     * @return self
     */
    public function addLeftJoin($field, $table)
    {
        $this->joins[self::JOIN_LEFT][$field] = $table;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasLeftJoins()
    {
        return !empty($this->joins[self::JOIN_LEFT]);
    }

    /**
     * @return array
     */
    public function getLeftJoins()
    {
        return $this->joins[self::JOIN_LEFT];
    }

    /**
     * @param string $field
     * @param string $table
     * @return self
     */
    public function addInnerJoin($field, $table)
    {
        $this->joins[self::JOIN_INNER][$field] = $table;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasInnerJoins()
    {
        return !empty($this->joins[self::JOIN_INNER]);
    }

    /**
     * @return array
     */
    public function getInnerJoins()
    {
        return $this->joins[self::JOIN_INNER];
    }

    /**
     * @return array
     */
    public function getGroupBy()
    {
        return $this->groupBy;
    }

    /**
     * @param array $exclude
     * @return self
     */
    public function setGroupBy(array $groupBy)
    {
        $this->groupBy = $groupBy;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasGroupBy()
    {
        return !empty($this->groupBy);
    }

    /**
     * Helper for date and time.
     *
     * @var mixed $input
     * @return \DateTime
     */
    protected function asDateTime($input)
    {
        if (is_string($input)) {
            return (new \DateTime($input));
        }

        return $input;
    }
}
