<?php
namespace Newland\Toubiz\Sync\Neos\Command;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\Service\ServiceFactory;
use Newland\Toubiz\Api\Service\Weather\AbstractWeatherService;
use Newland\Toubiz\Sync\Exception\UnknownDataProviderException;
use Newland\Toubiz\Sync\Neos\Importer\WeatherImporter;

/**
 * Weather command controller.
 *
 * Provides commands to manipulate weather data.
 *
 * @Flow\Scope("singleton")
 */
class WeatherCommandController extends AbstractCommandController
{
    /**
     * @Flow\InjectConfiguration
     * @var array
     */
    protected $configuration;

    /**
     * Synchronize command.
     *
     * Updates local weather data from API data source.
     *
     * @return void
     */
    public function synchronizeCommand()
    {
        $configurationKey = $this->getConfigurationKey();

        $configuration = $this->getConfigurationForService($configurationKey);
        if (!$configuration) {
            throw new \Exception('Meteotest/Api service not configured!');
        }

        /** @var AbstractWeatherService $service */
        $service = ServiceFactory::get($configurationKey);
        $service->setApiKey($configuration['apiKey']);
        $service->setParameters($configuration['parameters']);

        $forecast = $service->fetchForecast();
        $importer = new WeatherImporter();
        $importer->import($forecast);
    }

    /**
     * @return string
     */
    private function getConfigurationKey(): string
    {
        $configurationMap = [
            'meteotest' => 'Meteotest/Api',
            'openweathermap' => 'OpenWeatherMap/Api',
        ];
        try {
            $dataProvider = $this->configuration['dataProvider']['weather'];
            return $configurationMap[$dataProvider];
        } catch (\Exception $exception) {
            throw new UnknownDataProviderException('Unknown weather data provider or weather data provider not configured.');
        }
    }
}
