<?php declare(strict_types=1);

namespace Newland\Toubiz\Sync\Neos\Command\Task;

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\Constants\Language;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ServiceConstants;
use Newland\Toubiz\Api\ObjectAdapter\ServiceAdapterInterface;
use Newland\Toubiz\Api\Service\Tportal\ApiService;
use Newland\Toubiz\Sync\Neos\Command\Helper\ApiServiceHelper;
use Newland\Toubiz\Sync\Neos\Command\Helper\ConfigurationHelper;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ServiceRepository;
use Newland\Toubiz\Sync\Neos\Importer\ServiceImporter;
use Newland\Toubiz\Sync\Neos\Logging\LoggerFactory;
use Psr\Log\LoggerInterface;

class SynchronizePackagesFromTportal implements SynchronizationTask, DeletesOld
{

    /**
     * @var ApiServiceHelper
     * @Flow\Inject()
     */
    protected $apiServiceHelper;

    /**
     * @var ServiceImporter
     * @Flow\Inject()
     */
    protected $serviceImporter;

    /**
     * @var ServiceRepository
     * @Flow\Inject()
     */
    protected $serviceRepository;

    /** @var LoggerInterface */
    protected $logger;

    public function injectLogger(LoggerFactory $loggerFactory)
    {
        $this->logger = $loggerFactory->getLogger();
    }

    public function name(): string
    {
        return 'Packages';
    }

    public function configurations(ConfigurationHelper $configurationHelper): \Generator
    {
        $configuration = $configurationHelper->getConfigurationForService('Tportal/Api');
        foreach ($configuration['clients'] ?? [] as $key => $clientConfiguration) {
            $enable = $clientConfiguration['packages'] ?? null;
            if ($enable === null) {
                $this->logger->warning(sprintf(
                    'Configuration packages not set. Assuming `true`.' .
                    ' Please explicitly set `packages` to `true` or `false` for' .
                    ' `Newland.Toubiz.Sync.Neos.services.Tportal/Api.clients.%s.packages',
                    $key
                ));
            } elseif ($enable === false) {
                continue;
            }

            foreach ($clientConfiguration['languages'] ?? Language::values() as $language) {
                $clientConfiguration['language'] = $language;
                yield $configurationHelper->mergeWithDefaults($clientConfiguration);
            }
        }
    }

    public function synchronize(
        array $configuration,
        \Closure $errorHandlerWrapper,
        \Closure $onProgress
    ): SynchronizationResult {
        /** @var ApiService $service */
        $service = $this->apiServiceHelper->initializeApiServiceWithCommonConfigurationOptions(
            'Tportal/Api',
            $configuration
        );

        $touched = [];
        $this->serviceImporter->setLanguage($configuration['language']);
        $result = $service->fetchPackages(
            $errorHandlerWrapper(
                function (ServiceAdapterInterface $adapter, int $total) use ($onProgress, $configuration, &$touched) {
                    $adapter->setDetailUriTemplate($configuration['detailUri'] ?? null);
                    $onProgress(count($touched), $total);
                    $this->serviceImporter->setClient($configuration['client']);
                    $service = $this->serviceImporter->import($adapter);
                    if ($service) {
                        $touched[] = $service->getPersistenceObjectIdentifier();
                    }
                }
            )
        );

        return SynchronizationResult::fromServiceResult($result, $touched);
    }

    public function deleteOld(SynchronizationResult $result, array $configuration, \Closure $onProgress): void
    {
        $this->serviceRepository->removeBasedOnSynchronizationResult(
            $result,
            ServiceConstants::TYPE_PACKAGE,
            $configuration['client'],
            $configuration['language'],
            $onProgress
        );
    }
}
