<?php declare(strict_types=1);

namespace Newland\Toubiz\Sync\Neos\Importer;

use Neos\Flow\Log\PsrLoggerFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * This trait wraps a PSR logger interface and allows additional, predefined logging context
 * to be passed in to provide more context for log messages. It is built in a way to 100% wrap
 * a PSR logger: This trait implements the LoggerInterface itself.
 *
 * Additionally, this trait prepends a line break to every log message to ensure that the message
 * is printed on a new line, even when a progressbar is currently shown.
 */
trait LoggerWrapperWithPredefinedContext
{

    /** @var LoggerInterface */
    private $logger;
    abstract protected function loggerName(): string;
    public function injectLogger(PsrLoggerFactoryInterface $loggerFactory): void
    {
        $this->logger = $loggerFactory->get($this->loggerName());
    }

    protected $context = [];

    public function loggingContext(array $context): void
    {
        $this->context = $context;
    }

    protected function mergeLoggingContext(array $context): array
    {
        return array_merge($context, $this->context);
    }

    public function emergency($message, array $context = []): void
    {
        $this->logger->emergency("\n" . $message, $this->mergeLoggingContext($context));
    }

    public function alert($message, array $context = []): void
    {
        $this->logger->alert("\n" . $message, $this->mergeLoggingContext($context));
    }

    public function critical($message, array $context = []): void
    {
        $this->logger->critical("\n" . $message, $this->mergeLoggingContext($context));
    }

    public function error($message, array $context = []): void
    {
        $this->logger->error("\n" . $message, $this->mergeLoggingContext($context));
    }

    public function warning($message, array $context = []): void
    {
        $this->logger->warning("\n" . $message, $this->mergeLoggingContext($context));
    }

    public function notice($message, array $context = []): void
    {
        $this->logger->notice("\n" . $message, $this->mergeLoggingContext($context));
    }

    public function info($message, array $context = []): void
    {
        $this->logger->info("\n" . $message, $this->mergeLoggingContext($context));
    }

    public function debug($message, array $context = []): void
    {
        $this->logger->debug("\n" . $message, $this->mergeLoggingContext($context));
    }

    public function log($level, $message, array $context = []): void
    {
        $this->logger->log($level, "\n" . $message, $this->mergeLoggingContext($context));
    }
}
