<?php
namespace Newland\Toubiz\Sync\Neos\Command;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\AbstractQuery;
use Newland\Toubiz\Api\ObjectAdapter\PackageAdapterInterface;
use Newland\Toubiz\Api\Service\ServiceFactory;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\Service\Tportal\ApiService;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\PackageAdapter;
use Newland\Toubiz\Sync\Neos\Command\Helper\ApiServiceHelper;
use Newland\Toubiz\Sync\Neos\Domain\Model\Package;
use Newland\Toubiz\Sync\Neos\Domain\Repository\PackageRepository;
use Newland\Toubiz\Sync\Neos\Importer\PackageImporter;

class PackagesCommandController extends AbstractCommandController
{
    protected const TYPE_TPORTAL = 'tportal';

    /**
     * @var PackageImporter
     * @Flow\Inject()
     */
    protected $packageImporter;

    /**
     * @var PackageRepository
     * @Flow\Inject()
     */
    protected $packageRepository;

    /**
     * @var ApiServiceHelper
     * @Flow\Inject()
     */
    protected $apiServiceHelper;

    /**
     * @param bool $quiet
     * @param bool $finishHooks
     */
    public function synchronizeCommand($quiet = false, bool $finishHooks = true): void
    {
        if (!$quiet) {
            $this->showProgressOnCommandLine();
        }

        $configuration = $this->configurationHelper->getConfigurationForService('Tportal/Api');

        if ($configuration === null) {
            return;
        }

        foreach ($configuration['clients'] as $clientConfiguration) {
            foreach ($clientConfiguration['languages'] ?? $configuration['languages'] ?? [ 'de' ] as $language) {
                $this->synchronizePackagesFromTportal($clientConfiguration, $language);
            }
        }

        if ($finishHooks) {
            $this->objectPathMappingService->flushMappings(Package::class);
            $this->emitFinish();
        }
    }

    /**
     * Removes packages from the system according to the given clause.
     * If no WHERE clause is given then all packages will be deleted.
     *
     * # Remove all packages
     * $ php flow packages:remove
     *
     * # Remove packages according to WHERE clause
     * $ php flow packages:remove --where='package.client="untersee-portal"'
     *
     * # Remove single package
     * $ php flow packages:remove \
     *      --where="package.Persistence_Object_Identifier='a4625eb4-6e84-4834-969d-c9f1d447408b'"
     *
     *
     * @param string|null $where DQL WHERE clause selecting the packages to delete.
     * @param string|null $language Language to operate in. Operates in all languages if none specfied.
     */
    public function removeCommand(string $where = null, ?string $language = null)
    {
        if ($language === null) {
            $this->packageRepository->withoutLanguageHandling(
                function () use ($where) {
                    $this->remove($where);
                }
            );
        } else {
            $this->packageRepository->setLanguage($language);
            $this->remove($where);
        }
    }

    private function remove(?string $where): void
    {
        $query = $this->packageRepository->createQueryBuilder('package');
        if ($where) {
            $query->where($where);
            $this->outputLine('Deleting packages WHERE ' . $where);
        } else {
            $this->outputLine('Deleting all packages');
        }

        $count = (clone $query)->select('COUNT(package) AS count')
                ->getQuery()
                ->execute([], AbstractQuery::HYDRATE_ARRAY)[0]['count'] ?? 0;
        $this->askForConfirmationAndAbortIfNoneGiven(
            sprintf('Do you really want to remove %d packages?', $count)
        );

        $this->output->progressStart($count);
        foreach ($query->getQuery()->execute() as $package) {
            $this->packageRepository->remove($package);
            $this->output->progressAdvance();
        }

        $this->output->progressFinish();
    }

    private function synchronizePackagesFromTportal(array $configuration, string $language)
    {
        $client = $configuration['client'];
        $configuration['language'] = $language;
        $this->emitStart(static::TYPE_TPORTAL, compact('language', 'client'));

        /** @var \Newland\Toubiz\Api\Service\Tportal\ApiService $service */
        $service = $this->apiServiceHelper->initializeApiServiceWithCommonConfigurationOptions(
            'Tportal/Api',
            $configuration
        );

        $processed = 0;
        $service->fetchPackages(
            $this->wrapImportClosure(
                function (PackageAdapter $package) use ($client, &$processed, $language, $configuration) {
                    $package->setDetailUriTemplate($configuration['detailUri'] ?? null);
                    $this->emitProgress(
                        static::TYPE_TPORTAL,
                        ++$processed,
                        compact('language', 'client')
                    );
                    $this->packageImporter->setClient($client);
                    $this->packageImporter->import($package);
                }
            )
        );

        $this->emitEnd(static::TYPE_TPORTAL, compact('language', 'client'));
    }
}
