<?php declare(strict_types=1);
namespace Newland\Toubiz\Sync\Neos\Domain\Model;

use Neos\Flow\Annotations as Flow;
use Doctrine\ORM\Mapping as ORM;
use Newland\Toubiz\Sync\Neos\Domain\Model\Traits\HasDetailUri;
use Newland\Toubiz\Sync\Neos\Enum\Currency;
use Newland\Toubiz\Sync\Neos\Orm\Uuid\CustomUuidGeneration;
use Newland\Toubiz\Sync\Neos\Orm\Uuid\UuidGenerator;
use Newland\Toubiz\Sync\Neos\Translation\TranslatableEntity;
use Ramsey\Uuid\UuidInterface;

/**
 * @Flow\Entity
 * @ORM\Table(indexes={
 *      @ORM\Index(name="newland_toubiz_sync_neos_offer_import_ident", columns={"originalid"})
 * })
 */
class Offer extends AbstractEntity implements CustomUuidGeneration
{
    use TranslatableEntity,
        HasDetailUri;

    public function generateUuid(): UuidInterface
    {
        return UuidGenerator::uuidFromProperties([ $this->originalId, $this->language ]);
    }

    /** @var string */
    protected $name;

    /**
     * @var string|null
     * @ORM\Column(type="string", nullable=true)
     */
    protected $previewImage;

    /**
     * @var float|null
     * @ORM\Column(type="float", nullable=true)
     */
    protected $startingPrice;

    /**
     * ISO-4217 currency code.
     * Will be validated against the `Currency` enum in the setter.
     *
     * @var string|null
     * @ORM\Column(type="string", length=3, options={"fixed" = true}, nullable=true)
     */
    protected $startingPriceCurrency;

    /**
     * @var integer|null
     * @ORM\Column(type="integer", nullable=true)
     */
    protected $durationMinutes;

    /**
     * @var string|null
     * @ORM\Column(type="string", nullable=true)
     */
    protected $address;

    public function getName(): string
    {
        return $this->name;
    }

    public function setName(string $name): void
    {
        $this->name = $name;
    }

    public function getPreviewImage(): ?string
    {
        return $this->previewImage;
    }

    public function setPreviewImage(?string $previewImage): void
    {
        $this->previewImage = $previewImage;
    }

    public function getStartingPrice(): ?float
    {
        return $this->startingPrice;
    }

    public function setStartingPrice(?float $startingPrice): void
    {
        $this->startingPrice = $startingPrice;
    }

    public function getStartingPriceCurrency(): ?string
    {
        return $this->startingPriceCurrency;
    }

    public function setStartingPriceCurrency(?string $startingPriceCurrency): void
    {
        if ($startingPriceCurrency && !Currency::validValue($startingPriceCurrency)) {
            throw new \InvalidArgumentException(
                'The currency ' . $startingPriceCurrency . ' is undefined. ' .
                'Must be one of ' . implode(', ', Currency::values())
            );
        }
        $this->startingPriceCurrency = $startingPriceCurrency;
    }

    public function getDurationMinutes(): ?int
    {
        return $this->durationMinutes;
    }

    public function setDurationMinutes(?int $durationMinutes): void
    {
        $this->durationMinutes = $durationMinutes;
    }

    public function getAddress(): ?string
    {
        return $this->address;
    }

    public function setAddress(?string $address): void
    {
        $this->address = $address;
    }
}
