<?php declare(strict_types=1);
namespace Newland\Toubiz\Sync\Neos\Domain\Model;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\Mapping as ORM;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Sync\Neos\Domain\Model\Traits\PersistenceObjectIdentifier;
use Newland\Toubiz\Sync\Neos\Orm\Uuid\CustomUuidGeneration;
use Newland\Toubiz\Sync\Neos\Orm\Uuid\UuidGenerator;
use Ramsey\Uuid\UuidInterface;

/**
 * @Flow\Entity
 * @ORM\Table(uniqueConstraints={
 *      @ORM\UniqueConstraint(columns={"zipCode", "country"}, name="newland_toubiz_sync_neos_zipcode_unique")
 * })
 */
class ZipCode implements CustomUuidGeneration
{
    use PersistenceObjectIdentifier;

    /**
     * Includes valid zip code formats for most countries
     * @see http://www.fedex.com/be/shippingguide/postalcodeformat.html
     */
    public const VALID_PATTERN = '/^([\d]{4,8}|(\w\d\w\s\d\w\d)|(\w{2}\d{2}\w{2})'
    . '|(\w{2}\d\w\d\w{2})|(\w\d{2}\w{2})|(\w\d\w\d\w{2})|(\d{4}\-\d{3}))$/';

    /**
     * @ORM\Column(type="string", length=5)
     * @var string
     */
    protected $zipCode;
    /**
     * The country ISO code
     *
     * @ORM\Column(type="string", length=3, nullable=true)
     * @var string|null
     */
    protected $country;

    public function generateUuid(): UuidInterface
    {
        return UuidGenerator::uuidFromProperties([ $this->zipCode , $this->country]);
    }

    public function getZipCode(): string
    {
        return $this->zipCode;
    }

    public function setZipCode(string $zipCode): void
    {
        $this->zipCode = $zipCode;
    }

    public function getCountry(): ?string
    {
        return $this->country;
    }

    public function setCountry(?string $country = null): void
    {
        $this->country = $country;
    }
}
