<?php declare(strict_types=1);
namespace Newland\Toubiz\Sync\Neos\Command\Task;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\Constants\Language;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DbService;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DbService\PointOfInterestAdapter;
use Newland\Toubiz\Sync\Neos\Command\Helper\ApiServiceHelper;
use Newland\Toubiz\Sync\Neos\Command\Helper\ConfigurationHelper;
use Newland\Toubiz\Sync\Neos\Importer\ArticleImporter;
use Neos\Flow\Annotations as Flow;

class SynchronizePointOfInterestsFromDbService implements SynchronizationTask, ArticleSynchronizationTask
{
    use HandlesClientsForArticles;

    /**
     * @var ApiServiceHelper
     * @Flow\Inject()
     */
    protected $apiServiceHelper;

    /** @var int */
    private $processed = 0;

    /** @var array */
    private $ids = [];
    protected static $serviceName = 'Toubiz/Legacy/Db';

    public function name(): string
    {
        return 'POI';
    }

    public function configurations(ConfigurationHelper $configurationHelper): \Generator
    {
        $configuration = $configurationHelper->getConfigurationForService(static::$serviceName);
        if (!$configuration) {
            return;
        }

        foreach ($configuration['clients'] as $clientKey => $clientConfig) {
            $clientConfig['client'] = (string) ($clientConfig['client'] ?? $clientKey);
            foreach ($clientConfig['languages'] ?? $configuration['languages'] ?? [ Language::DE ] as $language) {
                $clientConfig['language'] = $language;
                yield $clientConfig;
            }
        }
    }

    public function synchronize(
        array $configuration,
        \Closure $errorHandlerWrapper,
        \Closure $onProgress
    ): SynchronizationResult {
        /** @var DbService $service */
        $service = $this->apiServiceHelper->initializeApiServiceWithCommonConfigurationOptions(
            static::$serviceName,
            $configuration
        );

        $this->processed = 0;
        $this->ids = [];
        $result = $service->fetchPointOfInterests(
            $errorHandlerWrapper(
                function (PointOfInterestAdapter $record, ?int $total) use ($onProgress, $configuration) {
                    $onProgress(++$this->processed, $total);

                    $importer = new ArticleImporter();
                    $importer->setLanguage($configuration['language']);
                    $importer->setClient($configuration['client']);
                    if ($record->getLevelOfMaintenance() > ($configuration['minLevelOfMaintenance'] ?? 0)) {
                        $this->ids[] = $importer->import($record)->getPersistenceObjectIdentifier();
                    } else {
                        $importer->remove($record);
                    }
                }
            )
        );

        return SynchronizationResult::fromServiceResult($result, $this->ids);
    }
}
