<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Repository;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\Query\Parameter;
use Doctrine\ORM\QueryBuilder;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Utility\Now;
use Neos\Flow\Persistence\QueryResultInterface;
use Neos\Flow\Validation\ValidatorResolver;
use Newland\Toubiz\Sync\Neos\Command\OrphanFinder;
use Newland\Toubiz\Sync\Neos\Domain\Filter\ArticleFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\FilterInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Translation\TranslatableRepository;
use Webit\DoctrineORM\QueryBuilder\Iterator\QueryBuilderIterator;

/**
 * Article repository.
 *
 * @Flow\Scope("singleton")
 *
 * @method Article|null findOneByOriginalId(string $originalId)
 * @method Article|null findOneBy(array $criteria, array $orderBy = null)
 * @method Article|null findByIdentifier(string $identifier)
 * @method QueryResultInterface|Article[] findByFilter(ArticleFilter $filter)
 */
class ArticleRepository extends AbstractRepository implements OrphanFinder
{
    use TranslatableRepository;

    const MIN_LATITUDE = -90;
    const MAX_LATITUDE = 90;
    const MIN_LONGITUDE = -180;
    const MAX_LONGITUDE = 180;

    /**
     * @var AttributeRepository
     * @Flow\Inject()
     */
    protected $attributeRepository;

    /**
     * @var ValidatorResolver
     * @Flow\Inject()
     */
    protected $validatorResolver;

    protected $alias = 'article';

    /**
     * Specific method for fetching all data for a data source
     * (e.g. multi-select in a neos node type property) in the
     * most performant way.
     *
     * @param ArticleFilter $filter
     * @return array
     */
    public function findAllForDataSource(ArticleFilter $filter)
    {
        $query = $this->createQueryBuilder('article');
        $query->select(
            [
                'article.name AS article_name',
                'article.Persistence_Object_Identifier AS article_identifier',
                'category.title AS category_title',

            ]
        )->leftJoin('article.categories', 'category')
            ->addOrderBy('category.title', 'asc')
            ->addOrderBy('article.name', 'asc');

        $this->articleClientFilterService->addClientWhereClause($query, $filter);

        return $query->getQuery()->getScalarResult();
    }

    /**
     * @param string $originalId
     * @param string $client
     * @return Article|null
     */
    public function findOneByOriginalIdAndClient(string $originalId, string $client)
    {
        $query = $this->createQueryBuilder('article');

        $result = $query
            ->andWhere($query->expr()->eq('article.originalId', ':originalId'))
            ->setParameter('originalId', $originalId)
            ->andWhere(
                $query->expr()->orX(
                    $query->expr()->eq('article.client', ':client'),
                    $query->expr()->eq('article.client', ':empty'),
                    $query->expr()->isNull('article.client')
                )
            )
            ->setParameter('client', $client)
            ->setParameter('empty', '')
            ->setMaxResults(1)
            ->getQuery()
            ->execute();

        return empty($result) ? null : $result[0];
    }

    /**
     * Applies filter functions (from the article filter) onto
     * the given query builder.
     *
     * @param ArticleFilter $filter
     * @param QueryBuilder $query
     * @return QueryBuilder
     */
    protected function applyFilter(FilterInterface $filter, QueryBuilder $query): QueryBuilder
    {
        $this->applyBasicFilter($filter, $query);

        // Category is always joined as e.g. sorting relies on it.
        $query->leftJoin('article.categories', 'category');
        $query->leftJoin('article.mainAddress', 'mainAddress');

        $mainType = $filter->getMainType();
        if ($mainType !== null) {
            $query->andWhere($query->expr()->eq('article.mainType', $mainType));
            if ($this->articleClientFilterService->needsFilter('mainType', $mainType, $filter)) {
                $clientWhere = $this->articleClientFilterService->generateEqExprFor(
                    $query,
                    'mainType',
                    $mainType,
                    $filter,
                    'article'
                );
                if ($clientWhere) {
                    $query->andWhere($clientWhere);
                }
            }
        } else {
            $this->articleClientFilterService->addClientWhereClause($query, $filter);
        }

        $categories = $filter->getCategories();
        if (!empty($categories)) {
            $field = 'category.' . $filter->getCategoriesIdentifierField();
            $query->andWhere($query->expr()->in($field, $categories));
        }

        $zips = $filter->getZips();
        if (!empty($zips)) {
            $query->andWhere(
                $query->expr()->in('mainAddress.zip', $zips)
            );
        }

        if ($filter->getExcludeUnsafeCoordinates()) {
            $query->andWhere(
                $query->expr()->isNotNull('mainAddress.latitude'),
                $query->expr()->isNotNull('mainAddress.longitude'),
                $query->expr()->lte('mainAddress.latitude', static::MAX_LATITUDE),
                $query->expr()->gte('mainAddress.latitude', static::MIN_LATITUDE),
                $query->expr()->lte('mainAddress.longitude', static::MAX_LONGITUDE),
                $query->expr()->gte('mainAddress.longitude', static::MIN_LONGITUDE),
                $query->expr()->neq('mainAddress.latitude', 0),
                $query->expr()->neq('mainAddress.longitude', 0)
            );
        }

        $identifiers = $filter->getIdentifiers();
        if (!empty($identifiers)) {
            $query->andWhere(
                $query->expr()->in('article.Persistence_Object_Identifier', $identifiers)
            );
        }

        $attributesIn = $filter->getAttributesIn();
        if (!empty($attributesIn)) {
            foreach ($attributesIn as $attribute => $values) {
                $subQuery = $this->attributeRepository->getArticlesUsingAttributesInQuery($attribute, $values);
                $query->andWhere($query->expr()->in('article', $subQuery->getDQL()));

                foreach ($subQuery->getParameters() as $parameter) {
                    /** @var Parameter $parameter */
                    $query->setParameter($parameter->getName(), $parameter->getValue());
                }
            }
        }

        $attributesRange = $filter->getAttributesRange();
        if (!empty($attributesRange)) {
            foreach ($attributesRange as $attribute => $config) {
                $subQuery = $this->attributeRepository
                    ->getArticlesUsingAttributesMinMaxQuery($attribute, $config['min'], $config['max']);
                $query->andWhere($query->expr()->in('article', $subQuery->getDQL()));

                foreach ($subQuery->getParameters() as $parameter) {
                    /** @var Parameter $parameter */
                    $query->setParameter($parameter->getName(), $parameter->getValue());
                }
            }
        }

        return $query;
    }

    public function deleteOldRecords(
        \DateInterval $timeToLive,
        int $mainType = null,
        \DateTimeImmutable $now = null
    ): int {
        $now = $now ?? new Now();
        $updatedAt = (clone $now)->sub($timeToLive);

        $query = $this->createQueryBuilder($this->getTableAlias());
        $query
            ->select()
            ->andWhere('article.updatedAt <= :updated_at')
            ->setParameter('updated_at', $updatedAt)
            ->orWhere($query->expr()->isNull('article.updatedAt'));

        if ($mainType !== null) {
            $query->andWhere($query->expr()->eq('article.mainType', $mainType));
        }

        $iterator = new QueryBuilderIterator($query);
        $count = 0;

        foreach ($iterator as $article) {
            $this->remove($article);
            $count++;

            if ($count % 20 === 0) {
                $this->entityManager->flush();
                $this->validatorResolver->reset();
            }
        }

        $this->persistenceManager->persistAll();

        return $count;
    }

    /**
     * Returns orphan objects (e.g. they are used nowhere and can safely be deleted
     * without any consequences).
     *
     * @return iterable
     */
    public function findOrphans()
    {
        $query = $this->createQueryBuilder('article');
        $query->andWhere($query->expr()->isNull('article.client'));
        return $query->getQuery()->execute();
    }
}
