<?php declare(strict_types=1);

namespace Newland\Toubiz\Sync\Neos\Tests\Unit\Domain\Repository;
use Neos\Flow\Tests\FunctionalTestCase;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;
use Newland\Toubiz\Sync\Neos\Tests\Factory\ArticleFactory;

class ArticleRepositoryTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;

    /** @var ArticleFactory */
    protected $factory;

    /** @var ArticleRepository */
    protected $subject;

    public function setUp(): void
    {
        parent::setUp();
        $this->factory = new ArticleFactory($this->objectManager);
        $this->subject = $this->objectManager->get(ArticleRepository::class);
    }

    public function testFindOneByOriginalIdAndClientRespectsLanguages(): void
    {
        $german = $this->factory->create([ 'language' => 'de', 'client' => 'foo', 'originalId' => 'bar' ]);
        $english = $this->factory->create([ 'language' => 'en', 'client' => 'foo', 'originalId' => 'bar' ]);

        $this->subject->setLanguage('de');
        $result = $this->subject->findOneByOriginalIdAndClient('bar', 'foo');
        $this->assertNotNull($result);
        $this->assertEquals($german->getPersistenceObjectIdentifier(), $result->getPersistenceObjectIdentifier());

        $this->subject->setLanguage('en');
        $result = $this->subject->findOneByOriginalIdAndClient('bar', 'foo');
        $this->assertNotNull($result);
        $this->assertEquals($english->getPersistenceObjectIdentifier(), $result->getPersistenceObjectIdentifier());
    }

    public function testDeleteOldRecordsRespectsLanguages(): void
    {
        $german = $this->factory->create([ 'language' => 'de', 'updatedAt' => new \DateTime('2019-01-01T13:00:00') ]);
        $english = $this->factory->create([ 'language' => 'en', 'updatedAt' => new \DateTime('2019-01-01T13:00:00') ]);

        $this->subject->setLanguage('de');
        $result = $this->subject->deleteOldRecords(new \DateInterval('PT2H'), null, new \DateTimeImmutable('2019-01-01T19:00:00'));
        $this->assertEquals(1, $result);

        $this->subject->setLanguage('de');
        $germanFromDatabase = $this->subject->findByIdentifier($german->getPersistenceObjectIdentifier());
        $this->assertNull($germanFromDatabase, 'German should have been deleted');

        $this->subject->setLanguage('en');
        $englishFromDatabase = $this->subject->findByIdentifier($english->getPersistenceObjectIdentifier());
        $this->assertNotNull($englishFromDatabase, 'English should have not been deleted');
    }

    public function testOrphanFindingRespectsLanguages(): void
    {
        // Both are made orphans by having no client
        $german = $this->factory->create([ 'language' => 'de' ]);
        $german->setClient(null);
        $this->subject->update($german);
        $english = $this->factory->create([ 'language' => 'en']);
        $english->setClient(null);
        $this->subject->update($english);
        $this->persistenceManager->persistAll();

        $this->subject->setLanguage('de');
        $result = $this->subject->findOrphans();
        $ids = array_map(
            function(Article $article) {
                return $article->getPersistenceObjectIdentifier();
            },
            (array) $result
        );

        $this->assertContains($german->getPersistenceObjectIdentifier(), $ids, 'Should have found german');
        $this->assertNotContains($english->getPersistenceObjectIdentifier(), $ids, 'Should not have found english');
    }

    public function testRemoveByIdsShouldRespectLanguages(): void
    {
        $german = $this->factory->create([ 'language' => 'de' ]);
        $english = $this->factory->create([ 'language' => 'en' ]);

        $this->subject->setLanguage('en');
        $this->subject->removeByIds([ $german->getPersistenceObjectIdentifier(), $english->getPersistenceObjectIdentifier() ]);

        $this->subject->setLanguage('de');
        $germanFromDatabase = $this->subject->findByIdentifier($german->getPersistenceObjectIdentifier());
        $this->assertNotNull($germanFromDatabase, 'German article should not have been deleted');

        $this->subject->setLanguage('en');
        $englishFromDatabase = $this->subject->findByIdentifier($english->getPersistenceObjectIdentifier());
        $this->assertNull($englishFromDatabase, 'English article should have been deleted');
    }
}
