<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Repository;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cache\CacheManager;
use Neos\Cache\Frontend\VariableFrontend;
use Newland\Toubiz\Sync\Neos\Domain\Model\Weather;

/**
 * Weather repository.
 *
 * @Flow\Scope("singleton")
 */
class WeatherRepository
{
    /**
     * @Flow\Inject
     * @var CacheManager
     */
    protected $cacheManager;

    /**
     * @var VariableFrontend
     */
    protected $cache;

    /**
     * Instance constructor.
     *
     * @return void
     */
    public function initializeObject()
    {
        $this->cache = $this->cacheManager->getCache('Newland_Toubiz_Sync_Neos-Weather');
    }

    /**
     * Find weather by city.
     *
     * @param string $location
     * @return Weather|null
     */
    public function findByCity($location)
    {
        $data = $this->cache->get('forecast');
        if ($data[$location]) {
            return $this->buildByDay($data[$location]);
        }
    }

    /**
     * Find all cities.
     *
     * @return array
     */
    public function findAllCities()
    {
        $data = $this->cache->get('forecast');
        if (!$data) {
            return [];
        }
        return array_keys($data);
    }

    /**
     * Stores data to cache.
     *
     * @param string identifier
     * @param mixed $data
     * @return void
     */
    public function store($identifier, $data)
    {
        $this->cache->set($identifier, $data);
    }

    /**
     * Build weather instances from given data by
     * splitting them up by day.
     *
     * @param array $data
     * @return array
     */
    protected function buildByDay(array $data)
    {
        $records = [];
        $currentDate = null;
        $dateRecords = [];
        foreach ($data as $dateString => $values) {
            $date = new \DateTime($dateString);
            if (!$currentDate) {
                $currentDate = $date;
            }

            if ($currentDate->format('Ymd') !== $date->format('Ymd')) {
                $records[] = (new Weather($dateRecords));
                $currentDate = null;
                $dateRecords = [];
            } else {
                $dateRecords[$dateString] = $values;
            }
        }

        return $records;
    }
}
