<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Repository;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\ObjectAdapter\Concern\TransportationFacilityConstants;
use Newland\Toubiz\Sync\Neos\Domain\Filter\TransportationFacilityFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Region;

/**
 * Transportation facility repository.
 *
 * @Flow\Scope("singleton")
 */
class TransportationFacilityRepository extends AbstractRepository
{
    /**
     * Find facilities matching given filter.
     *
     * @param TransportationFacilityFilter $filter
     * @return QueryResultInterface
     */
    public function findByFilter(TransportationFacilityFilter $filter)
    {
        $query = $this->createQueryBuilder('transportationFacility');
        $this->applyBasicFilter($filter, $query);
        $query->leftJoin('transportationFacility.regions', 'region');

        if ($filter->hasRegions()) {
            $query->andWhere(
                $query->expr()->in('region', $filter->getRegions())
            );
        }
        if ($filter->hasType()) {
            $query->andWhere(
                $query->expr()->in('transportationFacility.type', $filter->getTypes())
            );
        }

        return $query->getQuery()->execute();
    }

    /**
     * Calculates statistics for region.
     *
     * @param array $regions
     * @return array
     */
    public function getStatistics(array $regions = [])
    {
        $query = $this->createQueryBuilder('transportationFacility');
        $query->leftJoin('transportationFacility.regions', 'region');

        $constraint = '';
        if (!empty($regions)) {
            $regionIds = [];
            foreach ($regions as $region) {
                $regionIds[] = $region->getPersistenceObjectIdentifier();
            }
            $query->setParameter('regionIds', $regionIds);
            $constraint = 'AND region IN(:regionIds)';
        }

        $query->select('
            SUM (
                CASE WHEN
                    transportationFacility.type IN(
                        ' . TransportationFacilityConstants::TYPE_SKI_LIFT . ',
                        ' . TransportationFacilityConstants::TYPE_4_SEATER_CHAIRLIFT . ',
                        ' . TransportationFacilityConstants::TYPE_6_SEATER_CHAIRLIFT . ',
                        ' . TransportationFacilityConstants::TYPE_GONDOLA . '
                    )
                    AND transportationFacility.status = ' . TransportationFacilityConstants::STATUS_OPEN . '
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS lifts_open,
            SUM (
                CASE WHEN
                    transportationFacility.type IN(
                        ' . TransportationFacilityConstants::TYPE_SKI_LIFT . ',
                        ' . TransportationFacilityConstants::TYPE_4_SEATER_CHAIRLIFT . ',
                        ' . TransportationFacilityConstants::TYPE_6_SEATER_CHAIRLIFT . ',
                        ' . TransportationFacilityConstants::TYPE_GONDOLA . '
                    )
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS lifts_total,

            SUM (
                CASE WHEN
                    transportationFacility.type IN(
                        ' . TransportationFacilityConstants::TYPE_REVERSIBLE_AERIAL_TRAM . ',
                        ' . TransportationFacilityConstants::TYPE_COGWHEEL_RAILWAY . '
                    )
                    AND transportationFacility.status = ' . TransportationFacilityConstants::STATUS_OPEN . '
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS mountain_railways_open,
            SUM (
                CASE WHEN
                    transportationFacility.type IN(
                        ' . TransportationFacilityConstants::TYPE_REVERSIBLE_AERIAL_TRAM . ',
                        ' . TransportationFacilityConstants::TYPE_COGWHEEL_RAILWAY . '
                    )
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS mountain_railways_total
        ');
        return $query->getQuery()->execute()[0];
    }
}
